/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.adapter.enumerable;

import com.hazelcast.org.apache.calcite.plan.Convention;
import com.hazelcast.org.apache.calcite.plan.RelTraitSet;
import com.hazelcast.org.apache.calcite.rel.InvalidRelException;
import com.hazelcast.org.apache.calcite.rel.RelNode;
import com.hazelcast.org.apache.calcite.rel.convert.ConverterRule;
import com.hazelcast.org.apache.calcite.rel.logical.LogicalAggregate;

/**
 * Rule to convert a {@link com.hazelcast.org.apache.calcite.rel.logical.LogicalAggregate}
 * to an {@link EnumerableAggregate}.
 */
class EnumerableAggregateRule extends ConverterRule {
  EnumerableAggregateRule() {
    super(LogicalAggregate.class, Convention.NONE,
        EnumerableConvention.INSTANCE, "EnumerableAggregateRule");
  }

  public RelNode convert(RelNode rel) {
    final LogicalAggregate agg = (LogicalAggregate) rel;
    final RelTraitSet traitSet = rel.getCluster()
        .traitSet().replace(EnumerableConvention.INSTANCE);
    try {
      return new EnumerableAggregate(
          rel.getCluster(),
          traitSet,
          convert(agg.getInput(), traitSet),
          agg.getGroupSet(),
          agg.getGroupSets(),
          agg.getAggCallList());
    } catch (InvalidRelException e) {
      EnumerableRules.LOGGER.debug(e.toString());
      return null;
    }
  }
}
