/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.hazelcast.org.apache.hc.client5.http.protocol;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.hazelcast.org.apache.hc.client5.http.RouteInfo;
import com.hazelcast.org.apache.hc.client5.http.cookie.StandardCookieSpec;
import com.hazelcast.org.apache.hc.client5.http.config.RequestConfig;
import com.hazelcast.org.apache.hc.client5.http.cookie.Cookie;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieOrigin;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieSpec;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieSpecFactory;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieStore;
import com.hazelcast.org.apache.hc.core5.annotation.Contract;
import com.hazelcast.org.apache.hc.core5.annotation.ThreadingBehavior;
import com.hazelcast.org.apache.hc.core5.http.EntityDetails;
import com.hazelcast.org.apache.hc.core5.http.Header;
import com.hazelcast.org.apache.hc.core5.http.HttpException;
import com.hazelcast.org.apache.hc.core5.http.HttpRequest;
import com.hazelcast.org.apache.hc.core5.http.HttpRequestInterceptor;
import com.hazelcast.org.apache.hc.core5.http.Method;
import com.hazelcast.org.apache.hc.core5.http.config.Lookup;
import com.hazelcast.org.apache.hc.core5.http.protocol.HttpContext;
import com.hazelcast.org.apache.hc.core5.net.URIAuthority;
import com.hazelcast.org.apache.hc.core5.util.Args;
import com.hazelcast.org.apache.hc.core5.util.TextUtils;
import com.hazelcast.org.slf4j.Logger;
import com.hazelcast.org.slf4j.LoggerFactory;

/**
 * Request interceptor that matches cookies available in the current
 * {@link CookieStore} to the request being executed and generates
 * corresponding {@code Cookie} request headers.
 *
 * @since 4.0
 */
@Contract(threading = ThreadingBehavior.STATELESS)
public class RequestAddCookies implements HttpRequestInterceptor {

    private static final Logger LOG = LoggerFactory.getLogger(RequestAddCookies.class);

    public RequestAddCookies() {
        super();
    }

    @Override
    public void process(final HttpRequest request, final EntityDetails entity, final HttpContext context)
            throws HttpException, IOException {
        Args.notNull(request, "HTTP request");
        Args.notNull(context, "HTTP context");

        final String method = request.getMethod();
        if (Method.CONNECT.isSame(method) || Method.TRACE.isSame(method)) {
            return;
        }

        final HttpClientContext clientContext = HttpClientContext.adapt(context);
        final String exchangeId = clientContext.getExchangeId();

        // Obtain cookie store
        final CookieStore cookieStore = clientContext.getCookieStore();
        if (cookieStore == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} Cookie store not specified in HTTP context", exchangeId);
            }
            return;
        }

        // Obtain the registry of cookie specs
        final Lookup<CookieSpecFactory> registry = clientContext.getCookieSpecRegistry();
        if (registry == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} CookieSpec registry not specified in HTTP context", exchangeId);
            }
            return;
        }

        // Obtain the route (required)
        final RouteInfo route = clientContext.getHttpRoute();
        if (route == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} Connection route not set in the context", exchangeId);
            }
            return;
        }

        final RequestConfig config = clientContext.getRequestConfig();
        String cookieSpecName = config.getCookieSpec();
        if (cookieSpecName == null) {
            cookieSpecName = StandardCookieSpec.STRICT;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} Cookie spec selected: {}", exchangeId, cookieSpecName);
        }

        final URIAuthority authority = request.getAuthority();
        String path = request.getPath();
        if (TextUtils.isEmpty(path)) {
            path = "/";
        }
        String hostName = authority != null ? authority.getHostName() : null;
        if (hostName == null) {
            hostName = route.getTargetHost().getHostName();
        }
        int port = authority != null ? authority.getPort() : -1;
        if (port < 0) {
            port = route.getTargetHost().getPort();
        }
        final CookieOrigin cookieOrigin = new CookieOrigin(hostName, port, path, route.isSecure());

        // Get an instance of the selected cookie policy
        final CookieSpecFactory factory = registry.lookup(cookieSpecName);
        if (factory == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} Unsupported cookie spec: {}", exchangeId, cookieSpecName);
            }
            return;
        }
        final CookieSpec cookieSpec = factory.create(clientContext);
        // Get all cookies available in the HTTP state
        final List<Cookie> cookies = cookieStore.getCookies();
        // Find cookies matching the given origin
        final List<Cookie> matchedCookies = new ArrayList<>();
        final Date now = new Date();
        boolean expired = false;
        for (final Cookie cookie : cookies) {
            if (!cookie.isExpired(now)) {
                if (cookieSpec.match(cookie, cookieOrigin)) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("{} Cookie {} match {}", exchangeId, cookie, cookieOrigin);
                    }
                    matchedCookies.add(cookie);
                }
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} Cookie {} expired", exchangeId, cookie);
                }
                expired = true;
            }
        }
        // Per RFC 6265, 5.3
        // The user agent must evict all expired cookies if, at any time, an expired cookie
        // exists in the cookie store
        if (expired) {
            cookieStore.clearExpired(now);
        }
        // Generate Cookie request headers
        if (!matchedCookies.isEmpty()) {
            final List<Header> headers = cookieSpec.formatCookies(matchedCookies);
            for (final Header header : headers) {
                request.addHeader(header);
            }
        }

        // Stick the CookieSpec and CookieOrigin instances to the HTTP context
        // so they could be obtained by the response interceptor
        context.setAttribute(HttpClientContext.COOKIE_SPEC, cookieSpec);
        context.setAttribute(HttpClientContext.COOKIE_ORIGIN, cookieOrigin);
    }

}
