package com.nexacro.uiadapter17.jakarta.dao.mybatis;

import com.nexacro17.xapi.data.DataSet;
import com.nexacro17.xapi.tx.PlatformException;
import com.nexacro.uiadapter17.jakarta.core.data.NexacroFirstRowHandler;
import com.nexacro.uiadapter17.jakarta.core.data.convert.NexacroConvertException;
import com.nexacro.uiadapter17.jakarta.core.data.support.ObjectToDataSetConverter;
import com.nexacro.uiadapter17.jakarta.dao.NexacroFirstRowException;
import org.apache.ibatis.session.ResultContext;
import org.apache.ibatis.session.ResultHandler;

import java.util.Map;

/**
 * 
 * MyBatis를 사용하여 nexacro platform으로 대용량 데이터를 전송하려고 할때 사용되는 ResultHandler 이다.
 * iBatis의 SqlMapClient 기반의 SqlMapClientRowHandler를 MyBatis의 SqlSession 기반으로 대체한 클래스이다.
 * 기존 `SqlMapClientRowHandler`의 모든 기능을 MyBatis 환경에서 동일하게 제공하며, 대용량 데이터를 분할하여 전송하는 기능을 그대로 지원한다.
 * <p>아래와 같은 형식으로 queryWithRowHandler() 대신 select()로 처리한다.
 * 쿼리가 실행한 후 남아 있는 데이터가 존재할 수 있기 때문에 전송되지 않은 데이터를 전송한다.
 * 
 * <pre>
 *  // 기존 iBatis - SqlMapClient 방식
 *      SqlMapClientRowHandler rowHandler = new SqlMapClientRowHandler(firstRowHandler, sendName, firstRowCount);
 *      getSqlMapClientTemplate().queryWithRowHandler("largeDataDAO.selectLargeData", null, rowHandler);
 *      rowHandler.sendRemainData();
 *  // 신규 MyBatis - SqlSession 방식
 *      SqlSessionResultHandler rowHandler = new SqlSessionResultHandler(firstRowHandler, sendName, firstRowCount);
 *      getSqlSession().select("largeDataDAO.selectLargeData", null, rowHandler);
 *      rowHandler.sendRemainData();
 * </pre>
 * 
 * @author TechServ.
 * @since 11.06.2025
 * @version 1.0
 * @see
 */

public class SqlSessionResultHandler implements ResultHandler<Object> {

    private static final int DEFAULT_FIRSTROW_COUNT = 1000;
    
    private ObjectToDataSetConverter converter;
    private NexacroFirstRowHandler firstRowHandler;
    private String resultName;
    private int firstRowCount;
    
    private DataSet currentDataSet;
    private int currentCount = 0;

    /**
     * Constructs an instance of SqlSessionResultHandler.
     *
     * @param firstRowHandler the handler responsible for processing the first rows of the result;
     *                        it manages the data transfer and processing logic
     * @param resultName the name of the result to be used in creating the DataSet
     * @param firstRowCount the configuration value indicating how many rows are initially processed;
     *                      if the provided value is less than or equal to zero, a default value is used
     */
    public SqlSessionResultHandler(NexacroFirstRowHandler firstRowHandler, String resultName, int firstRowCount) {
        this.firstRowHandler = firstRowHandler;
        this.resultName = resultName;
        this.firstRowCount = firstRowCount;
        if(this.firstRowCount <= 0) {
            this.firstRowCount = DEFAULT_FIRSTROW_COUNT;
        }
        // TODO getting NexacroConverterFactory.getConverter();
        this.converter = new ObjectToDataSetConverter();
    }

    /**
     * Handles the processing of a result retrieved from a database query. This method processes
     * the result object, prepares the DataSet, adds rows to it, and determines when to send
     * a batch of rows for further processing. If any error occurs during the data preparation
     * or processing, the method throws a specific exception.
     *
     * @param resultContext the context containing the current result object from a database query,
     *                      which is processed and added to a DataSet
     * @throws NexacroFirstRowException if there is an error sending the data or converting
     *                                   an object into a DataSet
     */
    @Override
    public void handleResult(ResultContext<? extends Object> resultContext) {
        Object valueObject = resultContext.getResultObject();
        
        try {
            prepareDataSet(valueObject);
            addRow(valueObject);
            currentCount++;
            if(currentCount % firstRowCount == 0) {
                sendDataSet();
            }
        } catch (PlatformException e) {
            throw new NexacroFirstRowException("could not send data. e="+e.getMessage(), e);
        } catch (NexacroConvertException e) {
            throw new NexacroFirstRowException("object to dataset convert failed. e="+e.getMessage(), e);
        }
    }

    /**
     * 데이터 분할 전송 후 남아 있는 데이터를 전송한다.
     */
    public void sendRemainData() {
    	 // send remain data..
        DataSet remainDataSet = getDataSet();
        if(remainDataSet != null && remainDataSet.getRowCount() > 0) {
            try {
                firstRowHandler.sendDataSet(remainDataSet);
            } catch (PlatformException e) {
//                throw new NexacroException("could not send remain data. query="+queryId+" e="+e.getMessage(), e);
                throw new NexacroFirstRowException("could not send remain data. e="+e.getMessage(), e);
            }
        }
    }

    /**
     * Sends the current DataSet to the first row handler.
     * @throws PlatformException
     */
    private void sendDataSet() throws PlatformException {
        firstRowHandler.sendDataSet(currentDataSet);
    }

    /**
     * Adds a row of data into the current DataSet. The method determines the type of the provided
     * value object and delegates the addition of the row to the appropriate converter logic.
     *
     * @param valueObject the object representing the row data to be added; it can be a Map or any other object type
     * @throws NexacroConvertException if an error occurs during the conversion or row addition into the DataSet
     */
    private void addRow(Object valueObject) throws NexacroConvertException {
        if(valueObject instanceof Map) {
            converter.addRowIntoDataSet(currentDataSet, (Map) valueObject, false);
        } else {
            converter.addRowIntoDataSet(currentDataSet, valueObject);    
        }
        
    }

    /**
     * Prepares the DataSet object by initializing it if it is not already set.
     * The method adds columns into the DataSet based on the type of the provided value object.
     *
     * @param valueObject the object from which to prepare and populate the DataSet; it can be a Map or any other object type
     * @throws NexacroConvertException if an error occurs during the column addition or dataset preparation process
     */
    private void prepareDataSet(Object valueObject) throws NexacroConvertException {
        if(this.currentDataSet != null) {
            return;
        }
        this.currentDataSet = new DataSet(resultName != null? resultName: "RESULT0");
        
        if(valueObject instanceof Map) {
            converter.addColumnIntoDataSet(currentDataSet, (Map) valueObject);
        } else {
            converter.addColumnIntoDataSet(currentDataSet, valueObject);    
        }
    }

    /**
     * Gets the current DataSet object.
     * @return
     */
    public DataSet getDataSet() {
        return this.currentDataSet;
    }
}
