/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a CloudWatch metric of your choosing that can be used for predictive scaling.
 * </p>
 * <p>
 * For predictive scaling to work with a customized load metric specification, AWS Auto Scaling needs access to the
 * <code>Sum</code> and <code>Average</code> statistics that CloudWatch computes from metric data.
 * </p>
 * <p>
 * When you choose a load metric, make sure that the required <code>Sum</code> and <code>Average</code> statistics for
 * your metric are available in CloudWatch and that they provide relevant data for predictive scaling. The
 * <code>Sum</code> statistic must represent the total load on the resource, and the <code>Average</code> statistic must
 * represent the average load per capacity unit of the resource. For example, there is a metric that counts the number
 * of requests processed by your Auto Scaling group. If the <code>Sum</code> statistic represents the total request
 * count processed by the group, then the <code>Average</code> statistic for the specified metric must represent the
 * average request count processed by each instance of the group.
 * </p>
 * <p>
 * If you publish your own metrics, you can aggregate the data points at a given interval and then publish the
 * aggregated data points to CloudWatch. Before AWS Auto Scaling generates the forecast, it sums up all the metric data
 * points that occurred within each hour to match the granularity period that is used in the forecast (60 minutes).
 * </p>
 * <p>
 * For information about terminology, available metrics, or how to publish new metrics, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html">Amazon CloudWatch
 * Concepts</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p>
 * <p>
 * After creating your scaling plan, you can use the AWS Auto Scaling console to visualize forecasts for the specified
 * metric. For more information, see <a
 * href="https://docs.aws.amazon.com/autoscaling/plans/userguide/gs-create-scaling-plan.html#gs-view-resource">View
 * Scaling Information for a Resource</a> in the <i>AWS Auto Scaling User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomizedLoadMetricSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomizedLoadMetricSpecification.Builder, CustomizedLoadMetricSpecification> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(CustomizedLoadMetricSpecification::metricName))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(CustomizedLoadMetricSpecification::namespace))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<List<MetricDimension>> DIMENSIONS_FIELD = SdkField
            .<List<MetricDimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(CustomizedLoadMetricSpecification::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Statistic").getter(getter(CustomizedLoadMetricSpecification::statisticAsString))
            .setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistic").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Unit")
            .getter(getter(CustomizedLoadMetricSpecification::unit)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            NAMESPACE_FIELD, DIMENSIONS_FIELD, STATISTIC_FIELD, UNIT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final String namespace;

    private final List<MetricDimension> dimensions;

    private final String statistic;

    private final String unit;

    private CustomizedLoadMetricSpecification(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.dimensions = builder.dimensions;
        this.statistic = builder.statistic;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric.
     * </p>
     * 
     * @return The namespace of the metric.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dimensions of the metric.
     * </p>
     * <p>
     * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     * customized load metric specification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return The dimensions of the metric.</p>
     *         <p>
     *         Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     *         customized load metric specification.
     */
    public final List<MetricDimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The statistic of the metric. The only valid value is <code>Sum</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric. The only valid value is <code>Sum</code>.
     * @see MetricStatistic
     */
    public final MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic of the metric. The only valid value is <code>Sum</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric. The only valid value is <code>Sum</code>.
     * @see MetricStatistic
     */
    public final String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * 
     * @return The unit of the metric.
     */
    public final String unit() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomizedLoadMetricSpecification)) {
            return false;
        }
        CustomizedLoadMetricSpecification other = (CustomizedLoadMetricSpecification) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(statisticAsString(), other.statisticAsString()) && Objects.equals(unit(), other.unit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomizedLoadMetricSpecification").add("MetricName", metricName())
                .add("Namespace", namespace()).add("Dimensions", hasDimensions() ? dimensions() : null)
                .add("Statistic", statisticAsString()).add("Unit", unit()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomizedLoadMetricSpecification, T> g) {
        return obj -> g.apply((CustomizedLoadMetricSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomizedLoadMetricSpecification> {
        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your customized load metric specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your customized load metric specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.autoscalingplans.model.MetricDimension.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.autoscalingplans.model.MetricDimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.autoscalingplans.model.MetricDimension.Builder#build()} is called
         * immediately and its result is passed to {@link #dimensions(List<MetricDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.autoscalingplans.model.MetricDimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<MetricDimension>)
         */
        Builder dimensions(Consumer<MetricDimension.Builder>... dimensions);

        /**
         * <p>
         * The statistic of the metric. The only valid value is <code>Sum</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric. The only valid value is <code>Sum</code>.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic of the metric. The only valid value is <code>Sum</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric. The only valid value is <code>Sum</code>.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private String namespace;

        private List<MetricDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String statistic;

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomizedLoadMetricSpecification model) {
            metricName(model.metricName);
            namespace(model.namespace);
            dimensions(model.dimensions);
            statistic(model.statistic);
            unit(model.unit);
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final List<MetricDimension.Builder> getDimensions() {
            List<MetricDimension.Builder> result = MetricDimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<MetricDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> MetricDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatistic() {
            return statistic;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public CustomizedLoadMetricSpecification build() {
            return new CustomizedLoadMetricSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
