/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupstorage;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backupstorage.model.AccessDeniedException;
import software.amazon.awssdk.services.backupstorage.model.BackupStorageException;
import software.amazon.awssdk.services.backupstorage.model.BackupStorageRequest;
import software.amazon.awssdk.services.backupstorage.model.DataAlreadyExistsException;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.GetChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.GetChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataResponse;
import software.amazon.awssdk.services.backupstorage.model.IllegalArgumentException;
import software.amazon.awssdk.services.backupstorage.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.backupstorage.model.ListChunksRequest;
import software.amazon.awssdk.services.backupstorage.model.ListChunksResponse;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse;
import software.amazon.awssdk.services.backupstorage.model.NotReadableInputStreamException;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteResponse;
import software.amazon.awssdk.services.backupstorage.model.PutChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.PutChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.PutObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.PutObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupstorage.model.RetryableException;
import software.amazon.awssdk.services.backupstorage.model.ServiceInternalException;
import software.amazon.awssdk.services.backupstorage.model.ServiceUnavailableException;
import software.amazon.awssdk.services.backupstorage.model.StartObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.StartObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ThrottlingException;
import software.amazon.awssdk.services.backupstorage.transform.DeleteObjectRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.GetChunkRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.GetObjectMetadataRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.ListChunksRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.ListObjectsRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.NotifyObjectCompleteRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.PutChunkRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.PutObjectRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.StartObjectRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BackupStorageClient}.
 *
 * @see BackupStorageClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupStorageClient implements BackupStorageClient {
    private static final Logger log = Logger.loggerFor(DefaultBackupStorageClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final BackupStorageServiceClientConfiguration serviceClientConfiguration;

    protected DefaultBackupStorageClient(BackupStorageServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Delete Object from the incremental base Backup.
     *
     * @param deleteObjectRequest
     * @return Result of the DeleteObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteObjectResponse deleteObject(DeleteObjectRequest deleteObjectRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteObjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteObject");

            return clientHandler.execute(new ClientExecutionParams<DeleteObjectRequest, DeleteObjectResponse>()
                    .withOperationName("DeleteObject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteObjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetChunkResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Chunk data'.
     * @return The transformed result of the ResponseTransformer.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getChunk(GetChunkRequest getChunkRequest,
            ResponseTransformer<GetChunkResponse, ReturnT> responseTransformer) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetChunkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChunkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChunkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChunk");

            return clientHandler.execute(
                    new ClientExecutionParams<GetChunkRequest, GetChunkResponse>().withOperationName("GetChunk")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getChunkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetChunkRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetObjectMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Metadata blob.'.
     * @return The transformed result of the ResponseTransformer.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest,
            ResponseTransformer<GetObjectMetadataResponse, ReturnT> responseTransformer) throws ServiceUnavailableException,
            ServiceInternalException, ResourceNotFoundException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetObjectMetadataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetObjectMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getObjectMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetObjectMetadata");

            return clientHandler.execute(
                    new ClientExecutionParams<GetObjectMetadataRequest, GetObjectMetadataResponse>()
                            .withOperationName("GetObjectMetadata").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getObjectMetadataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetObjectMetadataRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * List chunks in a given Object
     *
     * @param listChunksRequest
     * @return Result of the ListChunks operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChunksResponse listChunks(ListChunksRequest listChunksRequest) throws ServiceUnavailableException,
            ResourceNotFoundException, ServiceInternalException, RetryableException, IllegalArgumentException,
            AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChunksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChunksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChunksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChunks");

            return clientHandler
                    .execute(new ClientExecutionParams<ListChunksRequest, ListChunksResponse>().withOperationName("ListChunks")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listChunksRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListChunksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * List all Objects in a given Backup.
     *
     * @param listObjectsRequest
     * @return Result of the ListObjects operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListObjectsResponse listObjects(ListObjectsRequest listObjectsRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ThrottlingException,
            ResourceNotFoundException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException,
            SdkClientException, BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListObjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListObjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListObjects");

            return clientHandler.execute(new ClientExecutionParams<ListObjectsRequest, ListObjectsResponse>()
                    .withOperationName("ListObjects").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listObjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListObjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Optional metadata associated with an
     *        Object. Maximum length is 4MB.'
     * @return Result of the NotifyObjectComplete operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public NotifyObjectCompleteResponse notifyObjectComplete(NotifyObjectCompleteRequest notifyObjectCompleteRequest,
            RequestBody requestBody) throws ServiceUnavailableException, ServiceInternalException,
            NotReadableInputStreamException, RetryableException, IllegalArgumentException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        notifyObjectCompleteRequest = applySignerOverride(notifyObjectCompleteRequest, Aws4UnsignedPayloadSigner.create());
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NotifyObjectCompleteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, NotifyObjectCompleteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyObjectCompleteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyObjectComplete");

            return clientHandler.execute(new ClientExecutionParams<NotifyObjectCompleteRequest, NotifyObjectCompleteResponse>()
                    .withOperationName("NotifyObjectComplete")
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withInput(notifyObjectCompleteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new NotifyObjectCompleteRequestMarshaller(protocolFactory))
                                    .requestBody(requestBody).transferEncoding(true).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Data to be uploaded'
     * @return Result of the PutChunk operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutChunkResponse putChunk(PutChunkRequest putChunkRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        putChunkRequest = applySignerOverride(putChunkRequest, Aws4UnsignedPayloadSigner.create());
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutChunkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutChunkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChunkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChunk");

            return clientHandler.execute(new ClientExecutionParams<PutChunkRequest, PutChunkResponse>()
                    .withOperationName("PutChunk")
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withInput(putChunkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new PutChunkRequestMarshaller(protocolFactory)).requestBody(requestBody)
                                    .transferEncoding(true).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Inline chunk data to be uploaded.'
     * @return Result of the PutObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutObjectResponse putObject(PutObjectRequest putObjectRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        putObjectRequest = applySignerOverride(putObjectRequest, Aws4UnsignedPayloadSigner.create());
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutObjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutObject");

            return clientHandler.execute(new ClientExecutionParams<PutObjectRequest, PutObjectResponse>()
                    .withOperationName("PutObject")
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withInput(putObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new PutObjectRequestMarshaller(protocolFactory)).requestBody(requestBody)
                                    .transferEncoding(true).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Start upload containing one or many chunks.
     *
     * @param startObjectRequest
     * @return Result of the StartObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws DataAlreadyExistsException
     *         Non-retryable exception. Attempted to create already existing object or chunk. This message contains a
     *         checksum of already presented data.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartObjectResponse startObject(StartObjectRequest startObjectRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ResourceNotFoundException,
            DataAlreadyExistsException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartObjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartObject");

            return clientHandler.execute(new ClientExecutionParams<StartObjectRequest, StartObjectResponse>()
                    .withOperationName("StartObject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartObjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends BackupStorageRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupStorageException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidKeyUsageException")
                                .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotReadableInputStreamException")
                                .exceptionBuilderSupplier(NotReadableInputStreamException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalArgumentException")
                                .exceptionBuilderSupplier(IllegalArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataAlreadyExistsException")
                                .exceptionBuilderSupplier(DataAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceInternalException")
                                .exceptionBuilderSupplier(ServiceInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RetryableException")
                                .exceptionBuilderSupplier(RetryableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public final BackupStorageServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
