/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsmv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a backup of an AWS CloudHSM cluster. All backup objects contain the <code>BackupId</code>,
 * <code>BackupState</code>, <code>ClusterId</code>, and <code>CreateTimestamp</code> parameters. Backups that were
 * copied into a destination region additionally contain the <code>CopyTimestamp</code>, <code>SourceBackup</code>,
 * <code>SourceCluster</code>, and <code>SourceRegion</code> parameters. A backup that is pending deletion will include
 * the <code>DeleteTimestamp</code> parameter.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Backup implements SdkPojo, Serializable, ToCopyableBuilder<Backup.Builder, Backup> {
    private static final SdkField<String> BACKUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupId").getter(getter(Backup::backupId)).setter(setter(Builder::backupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupId").build()).build();

    private static final SdkField<String> BACKUP_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupState").getter(getter(Backup::backupStateAsString)).setter(setter(Builder::backupState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupState").build()).build();

    private static final SdkField<String> CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterId").getter(getter(Backup::clusterId)).setter(setter(Builder::clusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterId").build()).build();

    private static final SdkField<Instant> CREATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTimestamp").getter(getter(Backup::createTimestamp)).setter(setter(Builder::createTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTimestamp").build()).build();

    private static final SdkField<Instant> COPY_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CopyTimestamp").getter(getter(Backup::copyTimestamp)).setter(setter(Builder::copyTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTimestamp").build()).build();

    private static final SdkField<Boolean> NEVER_EXPIRES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("NeverExpires").getter(getter(Backup::neverExpires)).setter(setter(Builder::neverExpires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NeverExpires").build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceRegion").getter(getter(Backup::sourceRegion)).setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion").build()).build();

    private static final SdkField<String> SOURCE_BACKUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceBackup").getter(getter(Backup::sourceBackup)).setter(setter(Builder::sourceBackup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceBackup").build()).build();

    private static final SdkField<String> SOURCE_CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceCluster").getter(getter(Backup::sourceCluster)).setter(setter(Builder::sourceCluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceCluster").build()).build();

    private static final SdkField<Instant> DELETE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeleteTimestamp").getter(getter(Backup::deleteTimestamp)).setter(setter(Builder::deleteTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteTimestamp").build()).build();

    private static final SdkField<List<Tag>> TAG_LIST_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("TagList")
            .getter(getter(Backup::tagList))
            .setter(setter(Builder::tagList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKUP_ID_FIELD,
            BACKUP_STATE_FIELD, CLUSTER_ID_FIELD, CREATE_TIMESTAMP_FIELD, COPY_TIMESTAMP_FIELD, NEVER_EXPIRES_FIELD,
            SOURCE_REGION_FIELD, SOURCE_BACKUP_FIELD, SOURCE_CLUSTER_FIELD, DELETE_TIMESTAMP_FIELD, TAG_LIST_FIELD));

    private static final long serialVersionUID = 1L;

    private final String backupId;

    private final String backupState;

    private final String clusterId;

    private final Instant createTimestamp;

    private final Instant copyTimestamp;

    private final Boolean neverExpires;

    private final String sourceRegion;

    private final String sourceBackup;

    private final String sourceCluster;

    private final Instant deleteTimestamp;

    private final List<Tag> tagList;

    private Backup(BuilderImpl builder) {
        this.backupId = builder.backupId;
        this.backupState = builder.backupState;
        this.clusterId = builder.clusterId;
        this.createTimestamp = builder.createTimestamp;
        this.copyTimestamp = builder.copyTimestamp;
        this.neverExpires = builder.neverExpires;
        this.sourceRegion = builder.sourceRegion;
        this.sourceBackup = builder.sourceBackup;
        this.sourceCluster = builder.sourceCluster;
        this.deleteTimestamp = builder.deleteTimestamp;
        this.tagList = builder.tagList;
    }

    /**
     * <p>
     * The identifier (ID) of the backup.
     * </p>
     * 
     * @return The identifier (ID) of the backup.
     */
    public final String backupId() {
        return backupId;
    }

    /**
     * <p>
     * The state of the backup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backupState} will
     * return {@link BackupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backupStateAsString}.
     * </p>
     * 
     * @return The state of the backup.
     * @see BackupState
     */
    public final BackupState backupState() {
        return BackupState.fromValue(backupState);
    }

    /**
     * <p>
     * The state of the backup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backupState} will
     * return {@link BackupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backupStateAsString}.
     * </p>
     * 
     * @return The state of the backup.
     * @see BackupState
     */
    public final String backupStateAsString() {
        return backupState;
    }

    /**
     * <p>
     * The identifier (ID) of the cluster that was backed up.
     * </p>
     * 
     * @return The identifier (ID) of the cluster that was backed up.
     */
    public final String clusterId() {
        return clusterId;
    }

    /**
     * <p>
     * The date and time when the backup was created.
     * </p>
     * 
     * @return The date and time when the backup was created.
     */
    public final Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * The date and time when the backup was copied from a source backup.
     * </p>
     * 
     * @return The date and time when the backup was copied from a source backup.
     */
    public final Instant copyTimestamp() {
        return copyTimestamp;
    }

    /**
     * <p>
     * Specifies whether the service should exempt a backup from the retention policy for the cluster. <code>True</code>
     * exempts a backup from the retention policy. <code>False</code> means the service applies the backup retention
     * policy defined at the cluster.
     * </p>
     * 
     * @return Specifies whether the service should exempt a backup from the retention policy for the cluster.
     *         <code>True</code> exempts a backup from the retention policy. <code>False</code> means the service
     *         applies the backup retention policy defined at the cluster.
     */
    public final Boolean neverExpires() {
        return neverExpires;
    }

    /**
     * <p>
     * The AWS Region that contains the source backup from which the new backup was copied.
     * </p>
     * 
     * @return The AWS Region that contains the source backup from which the new backup was copied.
     */
    public final String sourceRegion() {
        return sourceRegion;
    }

    /**
     * <p>
     * The identifier (ID) of the source backup from which the new backup was copied.
     * </p>
     * 
     * @return The identifier (ID) of the source backup from which the new backup was copied.
     */
    public final String sourceBackup() {
        return sourceBackup;
    }

    /**
     * <p>
     * The identifier (ID) of the cluster containing the source backup from which the new backup was copied.
     * </p>
     * 
     * @return The identifier (ID) of the cluster containing the source backup from which the new backup was copied.
     */
    public final String sourceCluster() {
        return sourceCluster;
    }

    /**
     * <p>
     * The date and time when the backup will be permanently deleted.
     * </p>
     * 
     * @return The date and time when the backup will be permanently deleted.
     */
    public final Instant deleteTimestamp() {
        return deleteTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagList property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTagList() {
        return tagList != null && !(tagList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags for the backup.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagList} method.
     * </p>
     * 
     * @return The list of tags for the backup.
     */
    public final List<Tag> tagList() {
        return tagList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(backupId());
        hashCode = 31 * hashCode + Objects.hashCode(backupStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(clusterId());
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(copyTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(neverExpires());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(sourceBackup());
        hashCode = 31 * hashCode + Objects.hashCode(sourceCluster());
        hashCode = 31 * hashCode + Objects.hashCode(deleteTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagList() ? tagList() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Backup)) {
            return false;
        }
        Backup other = (Backup) obj;
        return Objects.equals(backupId(), other.backupId()) && Objects.equals(backupStateAsString(), other.backupStateAsString())
                && Objects.equals(clusterId(), other.clusterId()) && Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(copyTimestamp(), other.copyTimestamp()) && Objects.equals(neverExpires(), other.neverExpires())
                && Objects.equals(sourceRegion(), other.sourceRegion()) && Objects.equals(sourceBackup(), other.sourceBackup())
                && Objects.equals(sourceCluster(), other.sourceCluster())
                && Objects.equals(deleteTimestamp(), other.deleteTimestamp()) && hasTagList() == other.hasTagList()
                && Objects.equals(tagList(), other.tagList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Backup").add("BackupId", backupId()).add("BackupState", backupStateAsString())
                .add("ClusterId", clusterId()).add("CreateTimestamp", createTimestamp()).add("CopyTimestamp", copyTimestamp())
                .add("NeverExpires", neverExpires()).add("SourceRegion", sourceRegion()).add("SourceBackup", sourceBackup())
                .add("SourceCluster", sourceCluster()).add("DeleteTimestamp", deleteTimestamp())
                .add("TagList", hasTagList() ? tagList() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BackupId":
            return Optional.ofNullable(clazz.cast(backupId()));
        case "BackupState":
            return Optional.ofNullable(clazz.cast(backupStateAsString()));
        case "ClusterId":
            return Optional.ofNullable(clazz.cast(clusterId()));
        case "CreateTimestamp":
            return Optional.ofNullable(clazz.cast(createTimestamp()));
        case "CopyTimestamp":
            return Optional.ofNullable(clazz.cast(copyTimestamp()));
        case "NeverExpires":
            return Optional.ofNullable(clazz.cast(neverExpires()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        case "SourceBackup":
            return Optional.ofNullable(clazz.cast(sourceBackup()));
        case "SourceCluster":
            return Optional.ofNullable(clazz.cast(sourceCluster()));
        case "DeleteTimestamp":
            return Optional.ofNullable(clazz.cast(deleteTimestamp()));
        case "TagList":
            return Optional.ofNullable(clazz.cast(tagList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Backup, T> g) {
        return obj -> g.apply((Backup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Backup> {
        /**
         * <p>
         * The identifier (ID) of the backup.
         * </p>
         * 
         * @param backupId
         *        The identifier (ID) of the backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupId(String backupId);

        /**
         * <p>
         * The state of the backup.
         * </p>
         * 
         * @param backupState
         *        The state of the backup.
         * @see BackupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupState
         */
        Builder backupState(String backupState);

        /**
         * <p>
         * The state of the backup.
         * </p>
         * 
         * @param backupState
         *        The state of the backup.
         * @see BackupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupState
         */
        Builder backupState(BackupState backupState);

        /**
         * <p>
         * The identifier (ID) of the cluster that was backed up.
         * </p>
         * 
         * @param clusterId
         *        The identifier (ID) of the cluster that was backed up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterId(String clusterId);

        /**
         * <p>
         * The date and time when the backup was created.
         * </p>
         * 
         * @param createTimestamp
         *        The date and time when the backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * The date and time when the backup was copied from a source backup.
         * </p>
         * 
         * @param copyTimestamp
         *        The date and time when the backup was copied from a source backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTimestamp(Instant copyTimestamp);

        /**
         * <p>
         * Specifies whether the service should exempt a backup from the retention policy for the cluster.
         * <code>True</code> exempts a backup from the retention policy. <code>False</code> means the service applies
         * the backup retention policy defined at the cluster.
         * </p>
         * 
         * @param neverExpires
         *        Specifies whether the service should exempt a backup from the retention policy for the cluster.
         *        <code>True</code> exempts a backup from the retention policy. <code>False</code> means the service
         *        applies the backup retention policy defined at the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder neverExpires(Boolean neverExpires);

        /**
         * <p>
         * The AWS Region that contains the source backup from which the new backup was copied.
         * </p>
         * 
         * @param sourceRegion
         *        The AWS Region that contains the source backup from which the new backup was copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        /**
         * <p>
         * The identifier (ID) of the source backup from which the new backup was copied.
         * </p>
         * 
         * @param sourceBackup
         *        The identifier (ID) of the source backup from which the new backup was copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceBackup(String sourceBackup);

        /**
         * <p>
         * The identifier (ID) of the cluster containing the source backup from which the new backup was copied.
         * </p>
         * 
         * @param sourceCluster
         *        The identifier (ID) of the cluster containing the source backup from which the new backup was copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceCluster(String sourceCluster);

        /**
         * <p>
         * The date and time when the backup will be permanently deleted.
         * </p>
         * 
         * @param deleteTimestamp
         *        The date and time when the backup will be permanently deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteTimestamp(Instant deleteTimestamp);

        /**
         * <p>
         * The list of tags for the backup.
         * </p>
         * 
         * @param tagList
         *        The list of tags for the backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Collection<Tag> tagList);

        /**
         * <p>
         * The list of tags for the backup.
         * </p>
         * 
         * @param tagList
         *        The list of tags for the backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Tag... tagList);

        /**
         * <p>
         * The list of tags for the backup.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudhsmv2.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.cloudhsmv2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudhsmv2.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tagList(List<Tag>)}.
         * 
         * @param tagList
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudhsmv2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagList(java.util.Collection<Tag>)
         */
        Builder tagList(Consumer<Tag.Builder>... tagList);
    }

    static final class BuilderImpl implements Builder {
        private String backupId;

        private String backupState;

        private String clusterId;

        private Instant createTimestamp;

        private Instant copyTimestamp;

        private Boolean neverExpires;

        private String sourceRegion;

        private String sourceBackup;

        private String sourceCluster;

        private Instant deleteTimestamp;

        private List<Tag> tagList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Backup model) {
            backupId(model.backupId);
            backupState(model.backupState);
            clusterId(model.clusterId);
            createTimestamp(model.createTimestamp);
            copyTimestamp(model.copyTimestamp);
            neverExpires(model.neverExpires);
            sourceRegion(model.sourceRegion);
            sourceBackup(model.sourceBackup);
            sourceCluster(model.sourceCluster);
            deleteTimestamp(model.deleteTimestamp);
            tagList(model.tagList);
        }

        public final String getBackupId() {
            return backupId;
        }

        public final void setBackupId(String backupId) {
            this.backupId = backupId;
        }

        @Override
        public final Builder backupId(String backupId) {
            this.backupId = backupId;
            return this;
        }

        public final String getBackupState() {
            return backupState;
        }

        public final void setBackupState(String backupState) {
            this.backupState = backupState;
        }

        @Override
        public final Builder backupState(String backupState) {
            this.backupState = backupState;
            return this;
        }

        @Override
        public final Builder backupState(BackupState backupState) {
            this.backupState(backupState == null ? null : backupState.toString());
            return this;
        }

        public final String getClusterId() {
            return clusterId;
        }

        public final void setClusterId(String clusterId) {
            this.clusterId = clusterId;
        }

        @Override
        public final Builder clusterId(String clusterId) {
            this.clusterId = clusterId;
            return this;
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        @Override
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final Instant getCopyTimestamp() {
            return copyTimestamp;
        }

        public final void setCopyTimestamp(Instant copyTimestamp) {
            this.copyTimestamp = copyTimestamp;
        }

        @Override
        public final Builder copyTimestamp(Instant copyTimestamp) {
            this.copyTimestamp = copyTimestamp;
            return this;
        }

        public final Boolean getNeverExpires() {
            return neverExpires;
        }

        public final void setNeverExpires(Boolean neverExpires) {
            this.neverExpires = neverExpires;
        }

        @Override
        public final Builder neverExpires(Boolean neverExpires) {
            this.neverExpires = neverExpires;
            return this;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        @Override
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        public final String getSourceBackup() {
            return sourceBackup;
        }

        public final void setSourceBackup(String sourceBackup) {
            this.sourceBackup = sourceBackup;
        }

        @Override
        public final Builder sourceBackup(String sourceBackup) {
            this.sourceBackup = sourceBackup;
            return this;
        }

        public final String getSourceCluster() {
            return sourceCluster;
        }

        public final void setSourceCluster(String sourceCluster) {
            this.sourceCluster = sourceCluster;
        }

        @Override
        public final Builder sourceCluster(String sourceCluster) {
            this.sourceCluster = sourceCluster;
            return this;
        }

        public final Instant getDeleteTimestamp() {
            return deleteTimestamp;
        }

        public final void setDeleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
        }

        @Override
        public final Builder deleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
            return this;
        }

        public final List<Tag.Builder> getTagList() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tagList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagList(Collection<Tag.BuilderImpl> tagList) {
            this.tagList = TagListCopier.copyFromBuilder(tagList);
        }

        @Override
        public final Builder tagList(Collection<Tag> tagList) {
            this.tagList = TagListCopier.copy(tagList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(Tag... tagList) {
            tagList(Arrays.asList(tagList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(Consumer<Tag.Builder>... tagList) {
            tagList(Stream.of(tagList).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Backup build() {
            return new Backup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
