/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a recommendation option for an Auto Scaling group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroupRecommendationOption implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingGroupRecommendationOption.Builder, AutoScalingGroupRecommendationOption> {
    private static final SdkField<AutoScalingGroupConfiguration> CONFIGURATION_FIELD = SdkField
            .<AutoScalingGroupConfiguration> builder(MarshallingType.SDK_POJO).memberName("configuration")
            .getter(getter(AutoScalingGroupRecommendationOption::configuration)).setter(setter(Builder::configuration))
            .constructor(AutoScalingGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<List<UtilizationMetric>> PROJECTED_UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("projectedUtilizationMetrics")
            .getter(getter(AutoScalingGroupRecommendationOption::projectedUtilizationMetrics))
            .setter(setter(Builder::projectedUtilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectedUtilizationMetrics")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> PERFORMANCE_RISK_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("performanceRisk").getter(getter(AutoScalingGroupRecommendationOption::performanceRisk))
            .setter(setter(Builder::performanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performanceRisk").build()).build();

    private static final SdkField<Integer> RANK_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("rank")
            .getter(getter(AutoScalingGroupRecommendationOption::rank)).setter(setter(Builder::rank))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rank").build()).build();

    private static final SdkField<SavingsOpportunity> SAVINGS_OPPORTUNITY_FIELD = SdkField
            .<SavingsOpportunity> builder(MarshallingType.SDK_POJO).memberName("savingsOpportunity")
            .getter(getter(AutoScalingGroupRecommendationOption::savingsOpportunity)).setter(setter(Builder::savingsOpportunity))
            .constructor(SavingsOpportunity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("savingsOpportunity").build())
            .build();

    private static final SdkField<String> MIGRATION_EFFORT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("migrationEffort").getter(getter(AutoScalingGroupRecommendationOption::migrationEffortAsString))
            .setter(setter(Builder::migrationEffort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("migrationEffort").build()).build();

    private static final SdkField<GpuInfo> INSTANCE_GPU_INFO_FIELD = SdkField.<GpuInfo> builder(MarshallingType.SDK_POJO)
            .memberName("instanceGpuInfo").getter(getter(AutoScalingGroupRecommendationOption::instanceGpuInfo))
            .setter(setter(Builder::instanceGpuInfo)).constructor(GpuInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceGpuInfo").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_FIELD,
            PROJECTED_UTILIZATION_METRICS_FIELD, PERFORMANCE_RISK_FIELD, RANK_FIELD, SAVINGS_OPPORTUNITY_FIELD,
            MIGRATION_EFFORT_FIELD, INSTANCE_GPU_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final AutoScalingGroupConfiguration configuration;

    private final List<UtilizationMetric> projectedUtilizationMetrics;

    private final Double performanceRisk;

    private final Integer rank;

    private final SavingsOpportunity savingsOpportunity;

    private final String migrationEffort;

    private final GpuInfo instanceGpuInfo;

    private AutoScalingGroupRecommendationOption(BuilderImpl builder) {
        this.configuration = builder.configuration;
        this.projectedUtilizationMetrics = builder.projectedUtilizationMetrics;
        this.performanceRisk = builder.performanceRisk;
        this.rank = builder.rank;
        this.savingsOpportunity = builder.savingsOpportunity;
        this.migrationEffort = builder.migrationEffort;
        this.instanceGpuInfo = builder.instanceGpuInfo;
    }

    /**
     * <p>
     * An array of objects that describe an Auto Scaling group configuration.
     * </p>
     * 
     * @return An array of objects that describe an Auto Scaling group configuration.
     */
    public final AutoScalingGroupConfiguration configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProjectedUtilizationMetrics property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasProjectedUtilizationMetrics() {
        return projectedUtilizationMetrics != null && !(projectedUtilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
     * option.
     * </p>
     * <note>
     * <p>
     * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
     * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
     * agent installed on them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProjectedUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the projected utilization metrics of the Auto Scaling group
     *         recommendation option.</p> <note>
     *         <p>
     *         The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
     *         Additionally, the <code>Memory</code> metric is returned only for resources that have the unified
     *         CloudWatch agent installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     */
    public final List<UtilizationMetric> projectedUtilizationMetrics() {
        return projectedUtilizationMetrics;
    }

    /**
     * <p>
     * The performance risk of the Auto Scaling group configuration recommendation.
     * </p>
     * <p>
     * Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs of your
     * workload. Compute Optimizer calculates an individual performance risk score for each specification of the
     * recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk throughput, disk IOPS, network
     * throughput, and network PPS. The performance risk of the recommended instance is calculated as the maximum
     * performance risk score across the analyzed resource specifications.
     * </p>
     * <p>
     * The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended resource
     * is predicted to always provide enough hardware capability. The higher the performance risk is, the more likely
     * you should validate whether the recommendation will meet the performance requirements of your workload before
     * migrating your resource.
     * </p>
     * 
     * @return The performance risk of the Auto Scaling group configuration recommendation.</p>
     *         <p>
     *         Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs
     *         of your workload. Compute Optimizer calculates an individual performance risk score for each
     *         specification of the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk
     *         throughput, disk IOPS, network throughput, and network PPS. The performance risk of the recommended
     *         instance is calculated as the maximum performance risk score across the analyzed resource specifications.
     *         </p>
     *         <p>
     *         The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended
     *         resource is predicted to always provide enough hardware capability. The higher the performance risk is,
     *         the more likely you should validate whether the recommendation will meet the performance requirements of
     *         your workload before migrating your resource.
     */
    public final Double performanceRisk() {
        return performanceRisk;
    }

    /**
     * <p>
     * The rank of the Auto Scaling group recommendation option.
     * </p>
     * <p>
     * The top recommendation option is ranked as <code>1</code>.
     * </p>
     * 
     * @return The rank of the Auto Scaling group recommendation option.</p>
     *         <p>
     *         The top recommendation option is ranked as <code>1</code>.
     */
    public final Integer rank() {
        return rank;
    }

    /**
     * <p>
     * An object that describes the savings opportunity for the Auto Scaling group recommendation option. Savings
     * opportunity includes the estimated monthly savings amount and percentage.
     * </p>
     * 
     * @return An object that describes the savings opportunity for the Auto Scaling group recommendation option.
     *         Savings opportunity includes the estimated monthly savings amount and percentage.
     */
    public final SavingsOpportunity savingsOpportunity() {
        return savingsOpportunity;
    }

    /**
     * <p>
     * The level of effort required to migrate from the current instance type to the recommended instance type.
     * </p>
     * <p>
     * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an Amazon
     * Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a workload
     * type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The migration effort
     * is <code>VeryLow</code> if both the current and recommended instance types are of the same CPU architecture.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #migrationEffort}
     * will return {@link MigrationEffort#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #migrationEffortAsString}.
     * </p>
     * 
     * @return The level of effort required to migrate from the current instance type to the recommended instance
     *         type.</p>
     *         <p>
     *         For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
     *         Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if
     *         a workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended.
     *         The migration effort is <code>VeryLow</code> if both the current and recommended instance types are of
     *         the same CPU architecture.
     * @see MigrationEffort
     */
    public final MigrationEffort migrationEffort() {
        return MigrationEffort.fromValue(migrationEffort);
    }

    /**
     * <p>
     * The level of effort required to migrate from the current instance type to the recommended instance type.
     * </p>
     * <p>
     * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an Amazon
     * Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a workload
     * type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The migration effort
     * is <code>VeryLow</code> if both the current and recommended instance types are of the same CPU architecture.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #migrationEffort}
     * will return {@link MigrationEffort#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #migrationEffortAsString}.
     * </p>
     * 
     * @return The level of effort required to migrate from the current instance type to the recommended instance
     *         type.</p>
     *         <p>
     *         For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
     *         Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if
     *         a workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended.
     *         The migration effort is <code>VeryLow</code> if both the current and recommended instance types are of
     *         the same CPU architecture.
     * @see MigrationEffort
     */
    public final String migrationEffortAsString() {
        return migrationEffort;
    }

    /**
     * <p>
     * Describes the GPU accelerator settings for the recommended instance type of the Auto Scaling group.
     * </p>
     * 
     * @return Describes the GPU accelerator settings for the recommended instance type of the Auto Scaling group.
     */
    public final GpuInfo instanceGpuInfo() {
        return instanceGpuInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(hasProjectedUtilizationMetrics() ? projectedUtilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(performanceRisk());
        hashCode = 31 * hashCode + Objects.hashCode(rank());
        hashCode = 31 * hashCode + Objects.hashCode(savingsOpportunity());
        hashCode = 31 * hashCode + Objects.hashCode(migrationEffortAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGpuInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroupRecommendationOption)) {
            return false;
        }
        AutoScalingGroupRecommendationOption other = (AutoScalingGroupRecommendationOption) obj;
        return Objects.equals(configuration(), other.configuration())
                && hasProjectedUtilizationMetrics() == other.hasProjectedUtilizationMetrics()
                && Objects.equals(projectedUtilizationMetrics(), other.projectedUtilizationMetrics())
                && Objects.equals(performanceRisk(), other.performanceRisk()) && Objects.equals(rank(), other.rank())
                && Objects.equals(savingsOpportunity(), other.savingsOpportunity())
                && Objects.equals(migrationEffortAsString(), other.migrationEffortAsString())
                && Objects.equals(instanceGpuInfo(), other.instanceGpuInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoScalingGroupRecommendationOption").add("Configuration", configuration())
                .add("ProjectedUtilizationMetrics", hasProjectedUtilizationMetrics() ? projectedUtilizationMetrics() : null)
                .add("PerformanceRisk", performanceRisk()).add("Rank", rank()).add("SavingsOpportunity", savingsOpportunity())
                .add("MigrationEffort", migrationEffortAsString()).add("InstanceGpuInfo", instanceGpuInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "projectedUtilizationMetrics":
            return Optional.ofNullable(clazz.cast(projectedUtilizationMetrics()));
        case "performanceRisk":
            return Optional.ofNullable(clazz.cast(performanceRisk()));
        case "rank":
            return Optional.ofNullable(clazz.cast(rank()));
        case "savingsOpportunity":
            return Optional.ofNullable(clazz.cast(savingsOpportunity()));
        case "migrationEffort":
            return Optional.ofNullable(clazz.cast(migrationEffortAsString()));
        case "instanceGpuInfo":
            return Optional.ofNullable(clazz.cast(instanceGpuInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingGroupRecommendationOption, T> g) {
        return obj -> g.apply((AutoScalingGroupRecommendationOption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingGroupRecommendationOption> {
        /**
         * <p>
         * An array of objects that describe an Auto Scaling group configuration.
         * </p>
         * 
         * @param configuration
         *        An array of objects that describe an Auto Scaling group configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(AutoScalingGroupConfiguration configuration);

        /**
         * <p>
         * An array of objects that describe an Auto Scaling group configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingGroupConfiguration.Builder}
         * avoiding the need to create one manually via {@link AutoScalingGroupConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoScalingGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(AutoScalingGroupConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link AutoScalingGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(AutoScalingGroupConfiguration)
         */
        default Builder configuration(Consumer<AutoScalingGroupConfiguration.Builder> configuration) {
            return configuration(AutoScalingGroupConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the Auto Scaling group
         *        recommendation option.</p> <note>
         *        <p>
         *        The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics
         *        returned. Additionally, the <code>Memory</code> metric is returned only for resources that have the
         *        unified CloudWatch agent installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the Auto Scaling group
         *        recommendation option.</p> <note>
         *        <p>
         *        The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics
         *        returned. Additionally, the <code>Memory</code> metric is returned only for resources that have the
         *        unified CloudWatch agent installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #projectedUtilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param projectedUtilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projectedUtilizationMetrics(java.util.Collection<UtilizationMetric>)
         */
        Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics);

        /**
         * <p>
         * The performance risk of the Auto Scaling group configuration recommendation.
         * </p>
         * <p>
         * Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs of
         * your workload. Compute Optimizer calculates an individual performance risk score for each specification of
         * the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk throughput, disk IOPS,
         * network throughput, and network PPS. The performance risk of the recommended instance is calculated as the
         * maximum performance risk score across the analyzed resource specifications.
         * </p>
         * <p>
         * The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended
         * resource is predicted to always provide enough hardware capability. The higher the performance risk is, the
         * more likely you should validate whether the recommendation will meet the performance requirements of your
         * workload before migrating your resource.
         * </p>
         * 
         * @param performanceRisk
         *        The performance risk of the Auto Scaling group configuration recommendation.</p>
         *        <p>
         *        Performance risk indicates the likelihood of the recommended instance type not meeting the resource
         *        needs of your workload. Compute Optimizer calculates an individual performance risk score for each
         *        specification of the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk
         *        throughput, disk IOPS, network throughput, and network PPS. The performance risk of the recommended
         *        instance is calculated as the maximum performance risk score across the analyzed resource
         *        specifications.
         *        </p>
         *        <p>
         *        The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the
         *        recommended resource is predicted to always provide enough hardware capability. The higher the
         *        performance risk is, the more likely you should validate whether the recommendation will meet the
         *        performance requirements of your workload before migrating your resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceRisk(Double performanceRisk);

        /**
         * <p>
         * The rank of the Auto Scaling group recommendation option.
         * </p>
         * <p>
         * The top recommendation option is ranked as <code>1</code>.
         * </p>
         * 
         * @param rank
         *        The rank of the Auto Scaling group recommendation option.</p>
         *        <p>
         *        The top recommendation option is ranked as <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rank(Integer rank);

        /**
         * <p>
         * An object that describes the savings opportunity for the Auto Scaling group recommendation option. Savings
         * opportunity includes the estimated monthly savings amount and percentage.
         * </p>
         * 
         * @param savingsOpportunity
         *        An object that describes the savings opportunity for the Auto Scaling group recommendation option.
         *        Savings opportunity includes the estimated monthly savings amount and percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder savingsOpportunity(SavingsOpportunity savingsOpportunity);

        /**
         * <p>
         * An object that describes the savings opportunity for the Auto Scaling group recommendation option. Savings
         * opportunity includes the estimated monthly savings amount and percentage.
         * </p>
         * This is a convenience method that creates an instance of the {@link SavingsOpportunity.Builder} avoiding the
         * need to create one manually via {@link SavingsOpportunity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SavingsOpportunity.Builder#build()} is called immediately and its
         * result is passed to {@link #savingsOpportunity(SavingsOpportunity)}.
         * 
         * @param savingsOpportunity
         *        a consumer that will call methods on {@link SavingsOpportunity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #savingsOpportunity(SavingsOpportunity)
         */
        default Builder savingsOpportunity(Consumer<SavingsOpportunity.Builder> savingsOpportunity) {
            return savingsOpportunity(SavingsOpportunity.builder().applyMutation(savingsOpportunity).build());
        }

        /**
         * <p>
         * The level of effort required to migrate from the current instance type to the recommended instance type.
         * </p>
         * <p>
         * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
         * Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a
         * workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The
         * migration effort is <code>VeryLow</code> if both the current and recommended instance types are of the same
         * CPU architecture.
         * </p>
         * 
         * @param migrationEffort
         *        The level of effort required to migrate from the current instance type to the recommended instance
         *        type.</p>
         *        <p>
         *        For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and
         *        an Amazon Web Services Graviton instance type is recommended. The migration effort is
         *        <code>Medium</code> if a workload type couldn't be inferred but an Amazon Web Services Graviton
         *        instance type is recommended. The migration effort is <code>VeryLow</code> if both the current and
         *        recommended instance types are of the same CPU architecture.
         * @see MigrationEffort
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationEffort
         */
        Builder migrationEffort(String migrationEffort);

        /**
         * <p>
         * The level of effort required to migrate from the current instance type to the recommended instance type.
         * </p>
         * <p>
         * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
         * Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a
         * workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The
         * migration effort is <code>VeryLow</code> if both the current and recommended instance types are of the same
         * CPU architecture.
         * </p>
         * 
         * @param migrationEffort
         *        The level of effort required to migrate from the current instance type to the recommended instance
         *        type.</p>
         *        <p>
         *        For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and
         *        an Amazon Web Services Graviton instance type is recommended. The migration effort is
         *        <code>Medium</code> if a workload type couldn't be inferred but an Amazon Web Services Graviton
         *        instance type is recommended. The migration effort is <code>VeryLow</code> if both the current and
         *        recommended instance types are of the same CPU architecture.
         * @see MigrationEffort
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationEffort
         */
        Builder migrationEffort(MigrationEffort migrationEffort);

        /**
         * <p>
         * Describes the GPU accelerator settings for the recommended instance type of the Auto Scaling group.
         * </p>
         * 
         * @param instanceGpuInfo
         *        Describes the GPU accelerator settings for the recommended instance type of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGpuInfo(GpuInfo instanceGpuInfo);

        /**
         * <p>
         * Describes the GPU accelerator settings for the recommended instance type of the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the {@link GpuInfo.Builder} avoiding the need to
         * create one manually via {@link GpuInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GpuInfo.Builder#build()} is called immediately and its result is
         * passed to {@link #instanceGpuInfo(GpuInfo)}.
         * 
         * @param instanceGpuInfo
         *        a consumer that will call methods on {@link GpuInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceGpuInfo(GpuInfo)
         */
        default Builder instanceGpuInfo(Consumer<GpuInfo.Builder> instanceGpuInfo) {
            return instanceGpuInfo(GpuInfo.builder().applyMutation(instanceGpuInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private AutoScalingGroupConfiguration configuration;

        private List<UtilizationMetric> projectedUtilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double performanceRisk;

        private Integer rank;

        private SavingsOpportunity savingsOpportunity;

        private String migrationEffort;

        private GpuInfo instanceGpuInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroupRecommendationOption model) {
            configuration(model.configuration);
            projectedUtilizationMetrics(model.projectedUtilizationMetrics);
            performanceRisk(model.performanceRisk);
            rank(model.rank);
            savingsOpportunity(model.savingsOpportunity);
            migrationEffort(model.migrationEffort);
            instanceGpuInfo(model.instanceGpuInfo);
        }

        public final AutoScalingGroupConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(AutoScalingGroupConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(AutoScalingGroupConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final List<UtilizationMetric.Builder> getProjectedUtilizationMetrics() {
            List<UtilizationMetric.Builder> result = ProjectedUtilizationMetricsCopier
                    .copyToBuilder(this.projectedUtilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProjectedUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copyFromBuilder(projectedUtilizationMetrics);
        }

        @Override
        public final Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copy(projectedUtilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Arrays.asList(projectedUtilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Stream.of(projectedUtilizationMetrics)
                    .map(c -> UtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Double getPerformanceRisk() {
            return performanceRisk;
        }

        public final void setPerformanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
        }

        @Override
        public final Builder performanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
            return this;
        }

        public final Integer getRank() {
            return rank;
        }

        public final void setRank(Integer rank) {
            this.rank = rank;
        }

        @Override
        public final Builder rank(Integer rank) {
            this.rank = rank;
            return this;
        }

        public final SavingsOpportunity.Builder getSavingsOpportunity() {
            return savingsOpportunity != null ? savingsOpportunity.toBuilder() : null;
        }

        public final void setSavingsOpportunity(SavingsOpportunity.BuilderImpl savingsOpportunity) {
            this.savingsOpportunity = savingsOpportunity != null ? savingsOpportunity.build() : null;
        }

        @Override
        public final Builder savingsOpportunity(SavingsOpportunity savingsOpportunity) {
            this.savingsOpportunity = savingsOpportunity;
            return this;
        }

        public final String getMigrationEffort() {
            return migrationEffort;
        }

        public final void setMigrationEffort(String migrationEffort) {
            this.migrationEffort = migrationEffort;
        }

        @Override
        public final Builder migrationEffort(String migrationEffort) {
            this.migrationEffort = migrationEffort;
            return this;
        }

        @Override
        public final Builder migrationEffort(MigrationEffort migrationEffort) {
            this.migrationEffort(migrationEffort == null ? null : migrationEffort.toString());
            return this;
        }

        public final GpuInfo.Builder getInstanceGpuInfo() {
            return instanceGpuInfo != null ? instanceGpuInfo.toBuilder() : null;
        }

        public final void setInstanceGpuInfo(GpuInfo.BuilderImpl instanceGpuInfo) {
            this.instanceGpuInfo = instanceGpuInfo != null ? instanceGpuInfo.build() : null;
        }

        @Override
        public final Builder instanceGpuInfo(GpuInfo instanceGpuInfo) {
            this.instanceGpuInfo = instanceGpuInfo;
            return this;
        }

        @Override
        public AutoScalingGroupRecommendationOption build() {
            return new AutoScalingGroupRecommendationOption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
