/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.entityresolution.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.entityresolution.EntityResolutionAsyncClient;
import software.amazon.awssdk.services.entityresolution.internal.UserAgentUtils;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse;
import software.amazon.awssdk.services.entityresolution.model.SchemaMappingSummary;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.entityresolution.EntityResolutionAsyncClient#listSchemaMappingsPaginator(software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of
 * {@link software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse} response pages as per
 * demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the subscribe helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.entityresolution.paginators.ListSchemaMappingsPublisher publisher = client.listSchemaMappingsPaginator(request);
 * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.entityresolution.paginators.ListSchemaMappingsPublisher publisher = client.listSchemaMappingsPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listSchemaMappings(software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListSchemaMappingsPublisher implements SdkPublisher<ListSchemaMappingsResponse> {
    private final EntityResolutionAsyncClient client;

    private final ListSchemaMappingsRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public ListSchemaMappingsPublisher(EntityResolutionAsyncClient client, ListSchemaMappingsRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private ListSchemaMappingsPublisher(EntityResolutionAsyncClient client, ListSchemaMappingsRequest firstRequest,
            boolean isLastPage) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new ListSchemaMappingsResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super ListSchemaMappingsResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<SchemaMappingSummary> schemaList() {
        Function<ListSchemaMappingsResponse, Iterator<SchemaMappingSummary>> getIterator = response -> {
            if (response != null && response.schemaList() != null) {
                return response.schemaList().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsPublisher.builder().nextPageFetcher(new ListSchemaMappingsResponseFetcher())
                .iteratorFunction(getIterator).isLastPage(isLastPage).build();
    }

    private class ListSchemaMappingsResponseFetcher implements AsyncPageFetcher<ListSchemaMappingsResponse> {
        @Override
        public boolean hasNextPage(final ListSchemaMappingsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<ListSchemaMappingsResponse> nextPage(final ListSchemaMappingsResponse previousPage) {
            if (previousPage == null) {
                return client.listSchemaMappings(firstRequest);
            }
            return client.listSchemaMappings(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
