/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For a custom routing accelerator, describes the port range and protocol for all endpoints (virtual private cloud
 * subnets) in an endpoint group to accept client traffic on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomRoutingDestinationDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomRoutingDestinationDescription.Builder, CustomRoutingDestinationDescription> {
    private static final SdkField<Integer> FROM_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FromPort").getter(getter(CustomRoutingDestinationDescription::fromPort))
            .setter(setter(Builder::fromPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromPort").build()).build();

    private static final SdkField<Integer> TO_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ToPort").getter(getter(CustomRoutingDestinationDescription::toPort)).setter(setter(Builder::toPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ToPort").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(CustomRoutingDestinationDescription::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FROM_PORT_FIELD,
            TO_PORT_FIELD, PROTOCOLS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer fromPort;

    private final Integer toPort;

    private final List<String> protocols;

    private CustomRoutingDestinationDescription(BuilderImpl builder) {
        this.fromPort = builder.fromPort;
        this.toPort = builder.toPort;
        this.protocols = builder.protocols;
    }

    /**
     * <p>
     * The first port, inclusive, in the range of ports for the endpoint group that is associated with a custom routing
     * accelerator.
     * </p>
     * 
     * @return The first port, inclusive, in the range of ports for the endpoint group that is associated with a custom
     *         routing accelerator.
     */
    public final Integer fromPort() {
        return fromPort;
    }

    /**
     * <p>
     * The last port, inclusive, in the range of ports for the endpoint group that is associated with a custom routing
     * accelerator.
     * </p>
     * 
     * @return The last port, inclusive, in the range of ports for the endpoint group that is associated with a custom
     *         routing accelerator.
     */
    public final Integer toPort() {
        return toPort;
    }

    /**
     * <p>
     * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
     * either TCP or UDP.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
     *         can be either TCP or UDP.
     */
    public final List<Protocol> protocols() {
        return ProtocolsCopier.copyStringToEnum(protocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
     * either TCP or UDP.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
     *         can be either TCP or UDP.
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fromPort());
        hashCode = 31 * hashCode + Objects.hashCode(toPort());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomRoutingDestinationDescription)) {
            return false;
        }
        CustomRoutingDestinationDescription other = (CustomRoutingDestinationDescription) obj;
        return Objects.equals(fromPort(), other.fromPort()) && Objects.equals(toPort(), other.toPort())
                && hasProtocols() == other.hasProtocols() && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomRoutingDestinationDescription").add("FromPort", fromPort()).add("ToPort", toPort())
                .add("Protocols", hasProtocols() ? protocolsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FromPort":
            return Optional.ofNullable(clazz.cast(fromPort()));
        case "ToPort":
            return Optional.ofNullable(clazz.cast(toPort()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomRoutingDestinationDescription, T> g) {
        return obj -> g.apply((CustomRoutingDestinationDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomRoutingDestinationDescription> {
        /**
         * <p>
         * The first port, inclusive, in the range of ports for the endpoint group that is associated with a custom
         * routing accelerator.
         * </p>
         * 
         * @param fromPort
         *        The first port, inclusive, in the range of ports for the endpoint group that is associated with a
         *        custom routing accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromPort(Integer fromPort);

        /**
         * <p>
         * The last port, inclusive, in the range of ports for the endpoint group that is associated with a custom
         * routing accelerator.
         * </p>
         * 
         * @param toPort
         *        The last port, inclusive, in the range of ports for the endpoint group that is associated with a
         *        custom routing accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toPort(Integer toPort);

        /**
         * <p>
         * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
         * either TCP or UDP.
         * </p>
         * 
         * @param protocols
         *        The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
         *        can be either TCP or UDP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
         * either TCP or UDP.
         * </p>
         * 
         * @param protocols
         *        The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
         *        can be either TCP or UDP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
         * either TCP or UDP.
         * </p>
         * 
         * @param protocols
         *        The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
         *        can be either TCP or UDP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Protocol> protocols);

        /**
         * <p>
         * The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol can be
         * either TCP or UDP.
         * </p>
         * 
         * @param protocols
         *        The protocol for the endpoint group that is associated with a custom routing accelerator. The protocol
         *        can be either TCP or UDP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Protocol... protocols);
    }

    static final class BuilderImpl implements Builder {
        private Integer fromPort;

        private Integer toPort;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomRoutingDestinationDescription model) {
            fromPort(model.fromPort);
            toPort(model.toPort);
            protocolsWithStrings(model.protocols);
        }

        public final Integer getFromPort() {
            return fromPort;
        }

        public final void setFromPort(Integer fromPort) {
            this.fromPort = fromPort;
        }

        @Override
        public final Builder fromPort(Integer fromPort) {
            this.fromPort = fromPort;
            return this;
        }

        public final Integer getToPort() {
            return toPort;
        }

        public final void setToPort(Integer toPort) {
            this.toPort = toPort;
        }

        @Override
        public final Builder toPort(Integer toPort) {
            this.toPort = toPort;
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<Protocol> protocols) {
            this.protocols = ProtocolsCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(Protocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        @Override
        public CustomRoutingDestinationDescription build() {
            return new CustomRoutingDestinationDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
