/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.healthlake;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.healthlake.model.AccessDeniedException;
import software.amazon.awssdk.services.healthlake.model.ConflictException;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.HealthLakeException;
import software.amazon.awssdk.services.healthlake.model.InternalServerException;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.TagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.TagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ThrottlingException;
import software.amazon.awssdk.services.healthlake.model.UntagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.UntagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ValidationException;
import software.amazon.awssdk.services.healthlake.transform.CreateFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DeleteFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirDatastoresRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirExportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirImportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link HealthLakeAsyncClient}.
 *
 * @see HealthLakeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthLakeAsyncClient implements HealthLakeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultHealthLakeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final HealthLakeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultHealthLakeAsyncClient(HealthLakeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a data store that can ingest and export FHIR formatted data.
     * </p>
     *
     * @param createFhirDatastoreRequest
     * @return A Java Future containing the result of the CreateFHIRDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.CreateFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/CreateFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFhirDatastoreResponse> createFHIRDatastore(
            CreateFhirDatastoreRequest createFhirDatastoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFHIRDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFhirDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFhirDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFhirDatastoreRequest, CreateFhirDatastoreResponse>()
                            .withOperationName("CreateFHIRDatastore")
                            .withMarshaller(new CreateFhirDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFhirDatastoreRequest));
            CompletableFuture<CreateFhirDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a data store.
     * </p>
     *
     * @param deleteFhirDatastoreRequest
     * @return A Java Future containing the result of the DeleteFHIRDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>ConflictException The data store is in a transition state and the user requested action can not be
     *         performed.</li>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.DeleteFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DeleteFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFhirDatastoreResponse> deleteFHIRDatastore(
            DeleteFhirDatastoreRequest deleteFhirDatastoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFHIRDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFhirDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFhirDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFhirDatastoreRequest, DeleteFhirDatastoreResponse>()
                            .withOperationName("DeleteFHIRDatastore")
                            .withMarshaller(new DeleteFhirDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFhirDatastoreRequest));
            CompletableFuture<DeleteFhirDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the properties associated with the FHIR data store, including the data store ID, data store ARN, data store
     * name, data store status, when the data store was created, data store type version, and the data store's endpoint.
     * </p>
     *
     * @param describeFhirDatastoreRequest
     * @return A Java Future containing the result of the DescribeFHIRDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.DescribeFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFhirDatastoreResponse> describeFHIRDatastore(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFhirDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFhirDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFhirDatastoreRequest, DescribeFhirDatastoreResponse>()
                            .withOperationName("DescribeFHIRDatastore")
                            .withMarshaller(new DescribeFhirDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFhirDatastoreRequest));
            CompletableFuture<DescribeFhirDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the properties of a FHIR export job, including the ID, ARN, name, and the status of the job.
     * </p>
     *
     * @param describeFhirExportJobRequest
     * @return A Java Future containing the result of the DescribeFHIRExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.DescribeFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFhirExportJobResponse> describeFHIRExportJob(
            DescribeFhirExportJobRequest describeFhirExportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFhirExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFhirExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFhirExportJobRequest, DescribeFhirExportJobResponse>()
                            .withOperationName("DescribeFHIRExportJob")
                            .withMarshaller(new DescribeFhirExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFhirExportJobRequest));
            CompletableFuture<DescribeFhirExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the properties of a FHIR import job, including the ID, ARN, name, and the status of the job.
     * </p>
     *
     * @param describeFhirImportJobRequest
     * @return A Java Future containing the result of the DescribeFHIRImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.DescribeFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFhirImportJobResponse> describeFHIRImportJob(
            DescribeFhirImportJobRequest describeFhirImportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFhirImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFhirImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFhirImportJobRequest, DescribeFhirImportJobResponse>()
                            .withOperationName("DescribeFHIRImportJob")
                            .withMarshaller(new DescribeFhirImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFhirImportJobRequest));
            CompletableFuture<DescribeFhirImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all FHIR data stores that are in the user’s account, regardless of data store status.
     * </p>
     *
     * @param listFhirDatastoresRequest
     * @return A Java Future containing the result of the ListFHIRDatastores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.ListFHIRDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRDatastores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFhirDatastoresResponse> listFHIRDatastores(ListFhirDatastoresRequest listFhirDatastoresRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirDatastoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRDatastores");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFhirDatastoresResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFhirDatastoresResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFhirDatastoresResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFhirDatastoresRequest, ListFhirDatastoresResponse>()
                            .withOperationName("ListFHIRDatastores")
                            .withMarshaller(new ListFhirDatastoresRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFhirDatastoresRequest));
            CompletableFuture<ListFhirDatastoresResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all FHIR export jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirExportJobsRequest
     * @return A Java Future containing the result of the ListFHIRExportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.ListFHIRExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRExportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFhirExportJobsResponse> listFHIRExportJobs(ListFhirExportJobsRequest listFhirExportJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFhirExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFhirExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFhirExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFhirExportJobsRequest, ListFhirExportJobsResponse>()
                            .withOperationName("ListFHIRExportJobs")
                            .withMarshaller(new ListFhirExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFhirExportJobsRequest));
            CompletableFuture<ListFhirExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all FHIR import jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirImportJobsRequest
     * @return A Java Future containing the result of the ListFHIRImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.ListFHIRImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFhirImportJobsResponse> listFHIRImportJobs(ListFhirImportJobsRequest listFhirImportJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRImportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFhirImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFhirImportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFhirImportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFhirImportJobsRequest, ListFhirImportJobsResponse>()
                            .withOperationName("ListFHIRImportJobs")
                            .withMarshaller(new ListFhirImportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFhirImportJobsRequest));
            CompletableFuture<ListFhirImportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all existing tags associated with a data store.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Begins a FHIR export job.
     * </p>
     *
     * @param startFhirExportJobRequest
     * @return A Java Future containing the result of the StartFHIRExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.StartFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRExportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartFhirExportJobResponse> startFHIRExportJob(StartFhirExportJobRequest startFhirExportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartFhirExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFhirExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFhirExportJobRequest, StartFhirExportJobResponse>()
                            .withOperationName("StartFHIRExportJob")
                            .withMarshaller(new StartFhirExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startFhirExportJobRequest));
            CompletableFuture<StartFhirExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Begins a FHIR Import job.
     * </p>
     *
     * @param startFhirImportJobRequest
     * @return A Java Future containing the result of the StartFHIRImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ThrottlingException The user has exceeded their maximum number of allowed calls to the given API.</li>
     *         <li>AccessDeniedException Access is denied. Your account is not authorized to perform this operation.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>InternalServerException Unknown error occurs in the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.StartFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRImportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartFhirImportJobResponse> startFHIRImportJob(StartFhirImportJobRequest startFhirImportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartFhirImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFhirImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFhirImportJobRequest, StartFhirImportJobResponse>()
                            .withOperationName("StartFHIRImportJob")
                            .withMarshaller(new StartFhirImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startFhirImportJobRequest));
            CompletableFuture<StartFhirImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a user specified key and value tag to a data store.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a data store.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input parameter was invalid.</li>
     *         <li>ResourceNotFoundException The requested data store was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthLakeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final HealthLakeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HealthLakeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
