/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendraranking.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a document from a search service such as OpenSearch (self managed). Amazon Kendra Intelligent
 * Ranking uses this information to rank and score on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Document implements SdkPojo, Serializable, ToCopyableBuilder<Document.Builder, Document> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Document::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupId").getter(getter(Document::groupId)).setter(setter(Builder::groupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupId").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Document::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Body")
            .getter(getter(Document::body)).setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Body").build()).build();

    private static final SdkField<List<String>> TOKENIZED_TITLE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TokenizedTitle")
            .getter(getter(Document::tokenizedTitle))
            .setter(setter(Builder::tokenizedTitle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TokenizedTitle").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> TOKENIZED_BODY_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TokenizedBody")
            .getter(getter(Document::tokenizedBody))
            .setter(setter(Builder::tokenizedBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TokenizedBody").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> ORIGINAL_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("OriginalScore").getter(getter(Document::originalScore)).setter(setter(Builder::originalScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginalScore").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, GROUP_ID_FIELD,
            TITLE_FIELD, BODY_FIELD, TOKENIZED_TITLE_FIELD, TOKENIZED_BODY_FIELD, ORIGINAL_SCORE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String groupId;

    private final String title;

    private final String body;

    private final List<String> tokenizedTitle;

    private final List<String> tokenizedBody;

    private final Float originalScore;

    private Document(BuilderImpl builder) {
        this.id = builder.id;
        this.groupId = builder.groupId;
        this.title = builder.title;
        this.body = builder.body;
        this.tokenizedTitle = builder.tokenizedTitle;
        this.tokenizedBody = builder.tokenizedBody;
        this.originalScore = builder.originalScore;
    }

    /**
     * <p>
     * The identifier of the document from the search service.
     * </p>
     * 
     * @return The identifier of the document from the search service.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The optional group identifier of the document from the search service. Documents with the same group identifier
     * are grouped together and processed as one document within the service.
     * </p>
     * 
     * @return The optional group identifier of the document from the search service. Documents with the same group
     *         identifier are grouped together and processed as one document within the service.
     */
    public final String groupId() {
        return groupId;
    }

    /**
     * <p>
     * The title of the search service's document.
     * </p>
     * 
     * @return The title of the search service's document.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The body text of the search service's document.
     * </p>
     * 
     * @return The body text of the search service's document.
     */
    public final String body() {
        return body;
    }

    /**
     * For responses, this returns true if the service returned a value for the TokenizedTitle property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTokenizedTitle() {
        return tokenizedTitle != null && !(tokenizedTitle instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The title of the search service's document represented as a list of tokens or words. You must choose to provide
     * <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTokenizedTitle} method.
     * </p>
     * 
     * @return The title of the search service's document represented as a list of tokens or words. You must choose to
     *         provide <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
     */
    public final List<String> tokenizedTitle() {
        return tokenizedTitle;
    }

    /**
     * For responses, this returns true if the service returned a value for the TokenizedBody property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTokenizedBody() {
        return tokenizedBody != null && !(tokenizedBody instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The body text of the search service's document represented as a list of tokens or words. You must choose to
     * provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTokenizedBody} method.
     * </p>
     * 
     * @return The body text of the search service's document represented as a list of tokens or words. You must choose
     *         to provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
     */
    public final List<String> tokenizedBody() {
        return tokenizedBody;
    }

    /**
     * <p>
     * The original document score or rank from the search service. Amazon Kendra Intelligent Ranking gives the document
     * a new score or rank based on its intelligent search algorithms.
     * </p>
     * 
     * @return The original document score or rank from the search service. Amazon Kendra Intelligent Ranking gives the
     *         document a new score or rank based on its intelligent search algorithms.
     */
    public final Float originalScore() {
        return originalScore;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(groupId());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(hasTokenizedTitle() ? tokenizedTitle() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTokenizedBody() ? tokenizedBody() : null);
        hashCode = 31 * hashCode + Objects.hashCode(originalScore());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Document)) {
            return false;
        }
        Document other = (Document) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(groupId(), other.groupId())
                && Objects.equals(title(), other.title()) && Objects.equals(body(), other.body())
                && hasTokenizedTitle() == other.hasTokenizedTitle() && Objects.equals(tokenizedTitle(), other.tokenizedTitle())
                && hasTokenizedBody() == other.hasTokenizedBody() && Objects.equals(tokenizedBody(), other.tokenizedBody())
                && Objects.equals(originalScore(), other.originalScore());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Document").add("Id", id()).add("GroupId", groupId()).add("Title", title()).add("Body", body())
                .add("TokenizedTitle", hasTokenizedTitle() ? tokenizedTitle() : null)
                .add("TokenizedBody", hasTokenizedBody() ? tokenizedBody() : null).add("OriginalScore", originalScore()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "GroupId":
            return Optional.ofNullable(clazz.cast(groupId()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Body":
            return Optional.ofNullable(clazz.cast(body()));
        case "TokenizedTitle":
            return Optional.ofNullable(clazz.cast(tokenizedTitle()));
        case "TokenizedBody":
            return Optional.ofNullable(clazz.cast(tokenizedBody()));
        case "OriginalScore":
            return Optional.ofNullable(clazz.cast(originalScore()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Document, T> g) {
        return obj -> g.apply((Document) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Document> {
        /**
         * <p>
         * The identifier of the document from the search service.
         * </p>
         * 
         * @param id
         *        The identifier of the document from the search service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The optional group identifier of the document from the search service. Documents with the same group
         * identifier are grouped together and processed as one document within the service.
         * </p>
         * 
         * @param groupId
         *        The optional group identifier of the document from the search service. Documents with the same group
         *        identifier are grouped together and processed as one document within the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupId(String groupId);

        /**
         * <p>
         * The title of the search service's document.
         * </p>
         * 
         * @param title
         *        The title of the search service's document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The body text of the search service's document.
         * </p>
         * 
         * @param body
         *        The body text of the search service's document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * The title of the search service's document represented as a list of tokens or words. You must choose to
         * provide <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
         * </p>
         * 
         * @param tokenizedTitle
         *        The title of the search service's document represented as a list of tokens or words. You must choose
         *        to provide <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenizedTitle(Collection<String> tokenizedTitle);

        /**
         * <p>
         * The title of the search service's document represented as a list of tokens or words. You must choose to
         * provide <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
         * </p>
         * 
         * @param tokenizedTitle
         *        The title of the search service's document represented as a list of tokens or words. You must choose
         *        to provide <code>Title</code> or <code>TokenizedTitle</code>. You cannot provide both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenizedTitle(String... tokenizedTitle);

        /**
         * <p>
         * The body text of the search service's document represented as a list of tokens or words. You must choose to
         * provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
         * </p>
         * 
         * @param tokenizedBody
         *        The body text of the search service's document represented as a list of tokens or words. You must
         *        choose to provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenizedBody(Collection<String> tokenizedBody);

        /**
         * <p>
         * The body text of the search service's document represented as a list of tokens or words. You must choose to
         * provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
         * </p>
         * 
         * @param tokenizedBody
         *        The body text of the search service's document represented as a list of tokens or words. You must
         *        choose to provide <code>Body</code> or <code>TokenizedBody</code>. You cannot provide both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenizedBody(String... tokenizedBody);

        /**
         * <p>
         * The original document score or rank from the search service. Amazon Kendra Intelligent Ranking gives the
         * document a new score or rank based on its intelligent search algorithms.
         * </p>
         * 
         * @param originalScore
         *        The original document score or rank from the search service. Amazon Kendra Intelligent Ranking gives
         *        the document a new score or rank based on its intelligent search algorithms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originalScore(Float originalScore);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String groupId;

        private String title;

        private String body;

        private List<String> tokenizedTitle = DefaultSdkAutoConstructList.getInstance();

        private List<String> tokenizedBody = DefaultSdkAutoConstructList.getInstance();

        private Float originalScore;

        private BuilderImpl() {
        }

        private BuilderImpl(Document model) {
            id(model.id);
            groupId(model.groupId);
            title(model.title);
            body(model.body);
            tokenizedTitle(model.tokenizedTitle);
            tokenizedBody(model.tokenizedBody);
            originalScore(model.originalScore);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getGroupId() {
            return groupId;
        }

        public final void setGroupId(String groupId) {
            this.groupId = groupId;
        }

        @Override
        public final Builder groupId(String groupId) {
            this.groupId = groupId;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getBody() {
            return body;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final Collection<String> getTokenizedTitle() {
            if (tokenizedTitle instanceof SdkAutoConstructList) {
                return null;
            }
            return tokenizedTitle;
        }

        public final void setTokenizedTitle(Collection<String> tokenizedTitle) {
            this.tokenizedTitle = TitleTokensListCopier.copy(tokenizedTitle);
        }

        @Override
        public final Builder tokenizedTitle(Collection<String> tokenizedTitle) {
            this.tokenizedTitle = TitleTokensListCopier.copy(tokenizedTitle);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tokenizedTitle(String... tokenizedTitle) {
            tokenizedTitle(Arrays.asList(tokenizedTitle));
            return this;
        }

        public final Collection<String> getTokenizedBody() {
            if (tokenizedBody instanceof SdkAutoConstructList) {
                return null;
            }
            return tokenizedBody;
        }

        public final void setTokenizedBody(Collection<String> tokenizedBody) {
            this.tokenizedBody = BodyTokensListCopier.copy(tokenizedBody);
        }

        @Override
        public final Builder tokenizedBody(Collection<String> tokenizedBody) {
            this.tokenizedBody = BodyTokensListCopier.copy(tokenizedBody);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tokenizedBody(String... tokenizedBody) {
            tokenizedBody(Arrays.asList(tokenizedBody));
            return this;
        }

        public final Float getOriginalScore() {
            return originalScore;
        }

        public final void setOriginalScore(Float originalScore) {
            this.originalScore = originalScore;
        }

        @Override
        public final Builder originalScore(Float originalScore) {
            this.originalScore = originalScore;
            return this;
        }

        @Override
        public Document build() {
            return new Document(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
