/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * API Restrictions on the allowed actions, resources, and referers for an API key resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApiKeyRestrictions implements SdkPojo, Serializable,
        ToCopyableBuilder<ApiKeyRestrictions.Builder, ApiKeyRestrictions> {
    private static final SdkField<List<String>> ALLOW_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowActions")
            .getter(getter(ApiKeyRestrictions::allowActions))
            .setter(setter(Builder::allowActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ALLOW_REFERERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowReferers")
            .getter(getter(ApiKeyRestrictions::allowReferers))
            .setter(setter(Builder::allowReferers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowReferers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ALLOW_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowResources")
            .getter(getter(ApiKeyRestrictions::allowResources))
            .setter(setter(Builder::allowResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOW_ACTIONS_FIELD,
            ALLOW_REFERERS_FIELD, ALLOW_RESOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> allowActions;

    private final List<String> allowReferers;

    private final List<String> allowResources;

    private ApiKeyRestrictions(BuilderImpl builder) {
        this.allowActions = builder.allowActions;
        this.allowReferers = builder.allowReferers;
        this.allowResources = builder.allowResources;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowActions() {
        return allowActions != null && !(allowActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of allowed actions that an API key resource grants permissions to perform. You must have at least one
     * action for each type of resource. For example, if you have a place resource, you must include at least one place
     * action.
     * </p>
     * <p>
     * The following are valid values for the actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Map actions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Place actions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GetPlace</code> - Allows finding a place by place ID.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Route actions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>geo:CalculateRoute</code> - Allows point to point routing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You must use these strings exactly. For example, to provide access to map rendering, the only valid action is
     * <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
     * <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use <code>["geo:SearchPlaceIndexFor*"]</code> - you
     * must list each of the Place actions separately.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowActions} method.
     * </p>
     * 
     * @return A list of allowed actions that an API key resource grants permissions to perform. You must have at least
     *         one action for each type of resource. For example, if you have a place resource, you must include at
     *         least one place action.</p>
     *         <p>
     *         The following are valid values for the actions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Map actions</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Place actions</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GetPlace</code> - Allows finding a place by place ID.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Route actions</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>geo:CalculateRoute</code> - Allows point to point routing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You must use these strings exactly. For example, to provide access to map rendering, the only valid
     *         action is <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
     *         <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use
     *         <code>["geo:SearchPlaceIndexFor*"]</code> - you must list each of the Place actions separately.
     *         </p>
     */
    public final List<String> allowActions() {
        return allowActions;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowReferers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowReferers() {
        return allowReferers != null && !(allowReferers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An optional list of allowed HTTP referers for which requests must originate from. Requests using this API key
     * from other domains will not be allowed.
     * </p>
     * <p>
     * Requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
     * <code>$\-._+!*`(),;/?:@=&amp;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL encoding
     * purposes.
     * </p>
     * </li>
     * <li>
     * <p>
     * May contain wildcard characters question mark (?) and asterisk (*).
     * </p>
     * <p>
     * Question mark (?) will replace any single character (including hexadecimal digits).
     * </p>
     * <p>
     * Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
     * </p>
     * </li>
     * <li>
     * <p>
     * No spaces allowed. For example, <code>https://example.com</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowReferers} method.
     * </p>
     * 
     * @return An optional list of allowed HTTP referers for which requests must originate from. Requests using this API
     *         key from other domains will not be allowed.</p>
     *         <p>
     *         Requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
     *         <code>$\-._+!*`(),;/?:@=&amp;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL
     *         encoding purposes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         May contain wildcard characters question mark (?) and asterisk (*).
     *         </p>
     *         <p>
     *         Question mark (?) will replace any single character (including hexadecimal digits).
     *         </p>
     *         <p>
     *         Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No spaces allowed. For example, <code>https://example.com</code>.
     *         </p>
     *         </li>
     */
    public final List<String> allowReferers() {
        return allowReferers;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowResources() {
        return allowResources != null && !(allowResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of allowed resource ARNs that a API key bearer can perform actions on.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the resource-id to
     * match multiple resources of the same type.
     * </p>
     * </li>
     * <li>
     * <p>
     * The resources must be in the same <code>partition</code>, <code>region</code>, and <code>account-id</code> as the
     * key that is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * Other than wildcards, you must include the full ARN, including the <code>arn</code>, <code>partition</code>,
     * <code>service</code>, <code>region</code>, <code>account-id</code> and <code>resource-id</code>, delimited by
     * colons (:).
     * </p>
     * </li>
     * <li>
     * <p>
     * No spaces allowed, even with wildcards. For example,
     * <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about ARN format, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowResources} method.
     * </p>
     * 
     * @return A list of allowed resource ARNs that a API key bearer can perform actions on.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the
     *         resource-id to match multiple resources of the same type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The resources must be in the same <code>partition</code>, <code>region</code>, and
     *         <code>account-id</code> as the key that is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Other than wildcards, you must include the full ARN, including the <code>arn</code>,
     *         <code>partition</code>, <code>service</code>, <code>region</code>, <code>account-id</code> and
     *         <code>resource-id</code>, delimited by colons (:).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No spaces allowed, even with wildcards. For example,
     *         <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about ARN format, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs)</a>.
     */
    public final List<String> allowResources() {
        return allowResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowActions() ? allowActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowReferers() ? allowReferers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowResources() ? allowResources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApiKeyRestrictions)) {
            return false;
        }
        ApiKeyRestrictions other = (ApiKeyRestrictions) obj;
        return hasAllowActions() == other.hasAllowActions() && Objects.equals(allowActions(), other.allowActions())
                && hasAllowReferers() == other.hasAllowReferers() && Objects.equals(allowReferers(), other.allowReferers())
                && hasAllowResources() == other.hasAllowResources() && Objects.equals(allowResources(), other.allowResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ApiKeyRestrictions").add("AllowActions", hasAllowActions() ? allowActions() : null)
                .add("AllowReferers", hasAllowReferers() ? allowReferers() : null)
                .add("AllowResources", hasAllowResources() ? allowResources() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllowActions":
            return Optional.ofNullable(clazz.cast(allowActions()));
        case "AllowReferers":
            return Optional.ofNullable(clazz.cast(allowReferers()));
        case "AllowResources":
            return Optional.ofNullable(clazz.cast(allowResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ApiKeyRestrictions, T> g) {
        return obj -> g.apply((ApiKeyRestrictions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ApiKeyRestrictions> {
        /**
         * <p>
         * A list of allowed actions that an API key resource grants permissions to perform. You must have at least one
         * action for each type of resource. For example, if you have a place resource, you must include at least one
         * place action.
         * </p>
         * <p>
         * The following are valid values for the actions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Map actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Place actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetPlace</code> - Allows finding a place by place ID.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Route actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:CalculateRoute</code> - Allows point to point routing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You must use these strings exactly. For example, to provide access to map rendering, the only valid action is
         * <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
         * <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use <code>["geo:SearchPlaceIndexFor*"]</code> -
         * you must list each of the Place actions separately.
         * </p>
         * </note>
         * 
         * @param allowActions
         *        A list of allowed actions that an API key resource grants permissions to perform. You must have at
         *        least one action for each type of resource. For example, if you have a place resource, you must
         *        include at least one place action.</p>
         *        <p>
         *        The following are valid values for the actions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Map actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Place actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetPlace</code> - Allows finding a place by place ID.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Route actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:CalculateRoute</code> - Allows point to point routing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You must use these strings exactly. For example, to provide access to map rendering, the only valid
         *        action is <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
         *        <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use
         *        <code>["geo:SearchPlaceIndexFor*"]</code> - you must list each of the Place actions separately.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowActions(Collection<String> allowActions);

        /**
         * <p>
         * A list of allowed actions that an API key resource grants permissions to perform. You must have at least one
         * action for each type of resource. For example, if you have a place resource, you must include at least one
         * place action.
         * </p>
         * <p>
         * The following are valid values for the actions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Map actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Place actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetPlace</code> - Allows finding a place by place ID.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Route actions</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>geo:CalculateRoute</code> - Allows point to point routing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You must use these strings exactly. For example, to provide access to map rendering, the only valid action is
         * <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
         * <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use <code>["geo:SearchPlaceIndexFor*"]</code> -
         * you must list each of the Place actions separately.
         * </p>
         * </note>
         * 
         * @param allowActions
         *        A list of allowed actions that an API key resource grants permissions to perform. You must have at
         *        least one action for each type of resource. For example, if you have a place resource, you must
         *        include at least one place action.</p>
         *        <p>
         *        The following are valid values for the actions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Map actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:GetMap*</code> - Allows all actions needed for map rendering.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Place actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForText</code> - Allows geocoding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForPosition</code> - Allows reverse geocoding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:SearchPlaceIndexForSuggestions</code> - Allows generating suggestions from text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetPlace</code> - Allows finding a place by place ID.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Route actions</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>geo:CalculateRoute</code> - Allows point to point routing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>geo:CalculateRouteMatrix</code> - Allows calculating a matrix of routes.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You must use these strings exactly. For example, to provide access to map rendering, the only valid
         *        action is <code>geo:GetMap*</code> as an input to the list. <code>["geo:GetMap*"]</code> is valid but
         *        <code>["geo:GetMapTile"]</code> is not. Similarly, you cannot use
         *        <code>["geo:SearchPlaceIndexFor*"]</code> - you must list each of the Place actions separately.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowActions(String... allowActions);

        /**
         * <p>
         * An optional list of allowed HTTP referers for which requests must originate from. Requests using this API key
         * from other domains will not be allowed.
         * </p>
         * <p>
         * Requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
         * <code>$\-._+!*`(),;/?:@=&amp;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL encoding
         * purposes.
         * </p>
         * </li>
         * <li>
         * <p>
         * May contain wildcard characters question mark (?) and asterisk (*).
         * </p>
         * <p>
         * Question mark (?) will replace any single character (including hexadecimal digits).
         * </p>
         * <p>
         * Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
         * </p>
         * </li>
         * <li>
         * <p>
         * No spaces allowed. For example, <code>https://example.com</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowReferers
         *        An optional list of allowed HTTP referers for which requests must originate from. Requests using this
         *        API key from other domains will not be allowed.</p>
         *        <p>
         *        Requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
         *        <code>$\-._+!*`(),;/?:@=&amp;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL
         *        encoding purposes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        May contain wildcard characters question mark (?) and asterisk (*).
         *        </p>
         *        <p>
         *        Question mark (?) will replace any single character (including hexadecimal digits).
         *        </p>
         *        <p>
         *        Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No spaces allowed. For example, <code>https://example.com</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowReferers(Collection<String> allowReferers);

        /**
         * <p>
         * An optional list of allowed HTTP referers for which requests must originate from. Requests using this API key
         * from other domains will not be allowed.
         * </p>
         * <p>
         * Requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
         * <code>$\-._+!*`(),;/?:@=&amp;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL encoding
         * purposes.
         * </p>
         * </li>
         * <li>
         * <p>
         * May contain wildcard characters question mark (?) and asterisk (*).
         * </p>
         * <p>
         * Question mark (?) will replace any single character (including hexadecimal digits).
         * </p>
         * <p>
         * Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
         * </p>
         * </li>
         * <li>
         * <p>
         * No spaces allowed. For example, <code>https://example.com</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowReferers
         *        An optional list of allowed HTTP referers for which requests must originate from. Requests using this
         *        API key from other domains will not be allowed.</p>
         *        <p>
         *        Requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only alphanumeric characters (A–Z, a–z, 0–9) or any symbols in this list
         *        <code>$\-._+!*`(),;/?:@=&amp;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        May contain a percent (%) if followed by 2 hexadecimal digits (A-F, a-f, 0-9); this is used for URL
         *        encoding purposes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        May contain wildcard characters question mark (?) and asterisk (*).
         *        </p>
         *        <p>
         *        Question mark (?) will replace any single character (including hexadecimal digits).
         *        </p>
         *        <p>
         *        Asterisk (*) will replace any multiple characters (including multiple hexadecimal digits).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No spaces allowed. For example, <code>https://example.com</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowReferers(String... allowReferers);

        /**
         * <p>
         * A list of allowed resource ARNs that a API key bearer can perform actions on.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the resource-id
         * to match multiple resources of the same type.
         * </p>
         * </li>
         * <li>
         * <p>
         * The resources must be in the same <code>partition</code>, <code>region</code>, and <code>account-id</code> as
         * the key that is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * Other than wildcards, you must include the full ARN, including the <code>arn</code>, <code>partition</code>,
         * <code>service</code>, <code>region</code>, <code>account-id</code> and <code>resource-id</code>, delimited by
         * colons (:).
         * </p>
         * </li>
         * <li>
         * <p>
         * No spaces allowed, even with wildcards. For example,
         * <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about ARN format, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs)</a>.
         * </p>
         * 
         * @param allowResources
         *        A list of allowed resource ARNs that a API key bearer can perform actions on.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the
         *        resource-id to match multiple resources of the same type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The resources must be in the same <code>partition</code>, <code>region</code>, and
         *        <code>account-id</code> as the key that is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Other than wildcards, you must include the full ARN, including the <code>arn</code>,
         *        <code>partition</code>, <code>service</code>, <code>region</code>, <code>account-id</code> and
         *        <code>resource-id</code>, delimited by colons (:).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No spaces allowed, even with wildcards. For example,
         *        <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about ARN format, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowResources(Collection<String> allowResources);

        /**
         * <p>
         * A list of allowed resource ARNs that a API key bearer can perform actions on.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the resource-id
         * to match multiple resources of the same type.
         * </p>
         * </li>
         * <li>
         * <p>
         * The resources must be in the same <code>partition</code>, <code>region</code>, and <code>account-id</code> as
         * the key that is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * Other than wildcards, you must include the full ARN, including the <code>arn</code>, <code>partition</code>,
         * <code>service</code>, <code>region</code>, <code>account-id</code> and <code>resource-id</code>, delimited by
         * colons (:).
         * </p>
         * </li>
         * <li>
         * <p>
         * No spaces allowed, even with wildcards. For example,
         * <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about ARN format, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs)</a>.
         * </p>
         * 
         * @param allowResources
         *        A list of allowed resource ARNs that a API key bearer can perform actions on.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The ARN must be the correct ARN for a map, place, or route ARN. You may include wildcards in the
         *        resource-id to match multiple resources of the same type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The resources must be in the same <code>partition</code>, <code>region</code>, and
         *        <code>account-id</code> as the key that is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Other than wildcards, you must include the full ARN, including the <code>arn</code>,
         *        <code>partition</code>, <code>service</code>, <code>region</code>, <code>account-id</code> and
         *        <code>resource-id</code>, delimited by colons (:).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No spaces allowed, even with wildcards. For example,
         *        <code>arn:aws:geo:region:<i>account-id</i>:map/ExampleMap*</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about ARN format, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowResources(String... allowResources);
    }

    static final class BuilderImpl implements Builder {
        private List<String> allowActions = DefaultSdkAutoConstructList.getInstance();

        private List<String> allowReferers = DefaultSdkAutoConstructList.getInstance();

        private List<String> allowResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ApiKeyRestrictions model) {
            allowActions(model.allowActions);
            allowReferers(model.allowReferers);
            allowResources(model.allowResources);
        }

        public final Collection<String> getAllowActions() {
            if (allowActions instanceof SdkAutoConstructList) {
                return null;
            }
            return allowActions;
        }

        public final void setAllowActions(Collection<String> allowActions) {
            this.allowActions = ApiKeyRestrictionsAllowActionsListCopier.copy(allowActions);
        }

        @Override
        public final Builder allowActions(Collection<String> allowActions) {
            this.allowActions = ApiKeyRestrictionsAllowActionsListCopier.copy(allowActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowActions(String... allowActions) {
            allowActions(Arrays.asList(allowActions));
            return this;
        }

        public final Collection<String> getAllowReferers() {
            if (allowReferers instanceof SdkAutoConstructList) {
                return null;
            }
            return allowReferers;
        }

        public final void setAllowReferers(Collection<String> allowReferers) {
            this.allowReferers = ApiKeyRestrictionsAllowReferersListCopier.copy(allowReferers);
        }

        @Override
        public final Builder allowReferers(Collection<String> allowReferers) {
            this.allowReferers = ApiKeyRestrictionsAllowReferersListCopier.copy(allowReferers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowReferers(String... allowReferers) {
            allowReferers(Arrays.asList(allowReferers));
            return this;
        }

        public final Collection<String> getAllowResources() {
            if (allowResources instanceof SdkAutoConstructList) {
                return null;
            }
            return allowResources;
        }

        public final void setAllowResources(Collection<String> allowResources) {
            this.allowResources = ApiKeyRestrictionsAllowResourcesListCopier.copy(allowResources);
        }

        @Override
        public final Builder allowResources(Collection<String> allowResources) {
            this.allowResources = ApiKeyRestrictionsAllowResourcesListCopier.copy(allowResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowResources(String... allowResources) {
            allowResources(Arrays.asList(allowResources));
            return this;
        }

        @Override
        public ApiKeyRestrictions build() {
            return new ApiKeyRestrictions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
