/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchainquery.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The container of the transaction output.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TransactionOutputItem implements SdkPojo, Serializable,
        ToCopyableBuilder<TransactionOutputItem.Builder, TransactionOutputItem> {
    private static final SdkField<String> TRANSACTION_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transactionHash").getter(getter(TransactionOutputItem::transactionHash))
            .setter(setter(Builder::transactionHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionHash").build()).build();

    private static final SdkField<String> NETWORK_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("network")
            .getter(getter(TransactionOutputItem::networkAsString)).setter(setter(Builder::network))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("network").build()).build();

    private static final SdkField<Instant> TRANSACTION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("transactionTimestamp").getter(getter(TransactionOutputItem::transactionTimestamp))
            .setter(setter(Builder::transactionTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSACTION_HASH_FIELD,
            NETWORK_FIELD, TRANSACTION_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transactionHash;

    private final String network;

    private final Instant transactionTimestamp;

    private TransactionOutputItem(BuilderImpl builder) {
        this.transactionHash = builder.transactionHash;
        this.network = builder.network;
        this.transactionTimestamp = builder.transactionTimestamp;
    }

    /**
     * <p>
     * The hash of the transaction. It is generated whenever a transaction is verified and added to the blockchain.
     * </p>
     * 
     * @return The hash of the transaction. It is generated whenever a transaction is verified and added to the
     *         blockchain.
     */
    public final String transactionHash() {
        return transactionHash;
    }

    /**
     * <p>
     * The blockchain network where the transaction occurred.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #network} will
     * return {@link QueryNetwork#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkAsString}.
     * </p>
     * 
     * @return The blockchain network where the transaction occurred.
     * @see QueryNetwork
     */
    public final QueryNetwork network() {
        return QueryNetwork.fromValue(network);
    }

    /**
     * <p>
     * The blockchain network where the transaction occurred.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #network} will
     * return {@link QueryNetwork#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkAsString}.
     * </p>
     * 
     * @return The blockchain network where the transaction occurred.
     * @see QueryNetwork
     */
    public final String networkAsString() {
        return network;
    }

    /**
     * <p>
     * The time when the transaction occurred.
     * </p>
     * 
     * @return The time when the transaction occurred.
     */
    public final Instant transactionTimestamp() {
        return transactionTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transactionHash());
        hashCode = 31 * hashCode + Objects.hashCode(networkAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transactionTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TransactionOutputItem)) {
            return false;
        }
        TransactionOutputItem other = (TransactionOutputItem) obj;
        return Objects.equals(transactionHash(), other.transactionHash())
                && Objects.equals(networkAsString(), other.networkAsString())
                && Objects.equals(transactionTimestamp(), other.transactionTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TransactionOutputItem").add("TransactionHash", transactionHash())
                .add("Network", networkAsString()).add("TransactionTimestamp", transactionTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "transactionHash":
            return Optional.ofNullable(clazz.cast(transactionHash()));
        case "network":
            return Optional.ofNullable(clazz.cast(networkAsString()));
        case "transactionTimestamp":
            return Optional.ofNullable(clazz.cast(transactionTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TransactionOutputItem, T> g) {
        return obj -> g.apply((TransactionOutputItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TransactionOutputItem> {
        /**
         * <p>
         * The hash of the transaction. It is generated whenever a transaction is verified and added to the blockchain.
         * </p>
         * 
         * @param transactionHash
         *        The hash of the transaction. It is generated whenever a transaction is verified and added to the
         *        blockchain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionHash(String transactionHash);

        /**
         * <p>
         * The blockchain network where the transaction occurred.
         * </p>
         * 
         * @param network
         *        The blockchain network where the transaction occurred.
         * @see QueryNetwork
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryNetwork
         */
        Builder network(String network);

        /**
         * <p>
         * The blockchain network where the transaction occurred.
         * </p>
         * 
         * @param network
         *        The blockchain network where the transaction occurred.
         * @see QueryNetwork
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryNetwork
         */
        Builder network(QueryNetwork network);

        /**
         * <p>
         * The time when the transaction occurred.
         * </p>
         * 
         * @param transactionTimestamp
         *        The time when the transaction occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionTimestamp(Instant transactionTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String transactionHash;

        private String network;

        private Instant transactionTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(TransactionOutputItem model) {
            transactionHash(model.transactionHash);
            network(model.network);
            transactionTimestamp(model.transactionTimestamp);
        }

        public final String getTransactionHash() {
            return transactionHash;
        }

        public final void setTransactionHash(String transactionHash) {
            this.transactionHash = transactionHash;
        }

        @Override
        public final Builder transactionHash(String transactionHash) {
            this.transactionHash = transactionHash;
            return this;
        }

        public final String getNetwork() {
            return network;
        }

        public final void setNetwork(String network) {
            this.network = network;
        }

        @Override
        public final Builder network(String network) {
            this.network = network;
            return this;
        }

        @Override
        public final Builder network(QueryNetwork network) {
            this.network(network == null ? null : network.toString());
            return this;
        }

        public final Instant getTransactionTimestamp() {
            return transactionTimestamp;
        }

        public final void setTransactionTimestamp(Instant transactionTimestamp) {
            this.transactionTimestamp = transactionTimestamp;
        }

        @Override
        public final Builder transactionTimestamp(Instant transactionTimestamp) {
            this.transactionTimestamp = transactionTimestamp;
            return this;
        }

        @Override
        public TransactionOutputItem build() {
            return new TransactionOutputItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
