/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Specify one or more Automated ABR rule types. Note: Force include and Allowed renditions are mutually exclusive.
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutomatedAbrRule implements SdkPojo, Serializable,
        ToCopyableBuilder<AutomatedAbrRule.Builder, AutomatedAbrRule> {
    private static final SdkField<List<AllowedRenditionSize>> ALLOWED_RENDITIONS_FIELD = SdkField
            .<List<AllowedRenditionSize>> builder(MarshallingType.LIST)
            .memberName("AllowedRenditions")
            .getter(getter(AutomatedAbrRule::allowedRenditions))
            .setter(setter(Builder::allowedRenditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowedRenditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AllowedRenditionSize> builder(MarshallingType.SDK_POJO)
                                            .constructor(AllowedRenditionSize::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ForceIncludeRenditionSize>> FORCE_INCLUDE_RENDITIONS_FIELD = SdkField
            .<List<ForceIncludeRenditionSize>> builder(MarshallingType.LIST)
            .memberName("ForceIncludeRenditions")
            .getter(getter(AutomatedAbrRule::forceIncludeRenditions))
            .setter(setter(Builder::forceIncludeRenditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("forceIncludeRenditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ForceIncludeRenditionSize> builder(MarshallingType.SDK_POJO)
                                            .constructor(ForceIncludeRenditionSize::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<MinBottomRenditionSize> MIN_BOTTOM_RENDITION_SIZE_FIELD = SdkField
            .<MinBottomRenditionSize> builder(MarshallingType.SDK_POJO).memberName("MinBottomRenditionSize")
            .getter(getter(AutomatedAbrRule::minBottomRenditionSize)).setter(setter(Builder::minBottomRenditionSize))
            .constructor(MinBottomRenditionSize::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minBottomRenditionSize").build())
            .build();

    private static final SdkField<MinTopRenditionSize> MIN_TOP_RENDITION_SIZE_FIELD = SdkField
            .<MinTopRenditionSize> builder(MarshallingType.SDK_POJO).memberName("MinTopRenditionSize")
            .getter(getter(AutomatedAbrRule::minTopRenditionSize)).setter(setter(Builder::minTopRenditionSize))
            .constructor(MinTopRenditionSize::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minTopRenditionSize").build())
            .build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(AutomatedAbrRule::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOWED_RENDITIONS_FIELD,
            FORCE_INCLUDE_RENDITIONS_FIELD, MIN_BOTTOM_RENDITION_SIZE_FIELD, MIN_TOP_RENDITION_SIZE_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<AllowedRenditionSize> allowedRenditions;

    private final List<ForceIncludeRenditionSize> forceIncludeRenditions;

    private final MinBottomRenditionSize minBottomRenditionSize;

    private final MinTopRenditionSize minTopRenditionSize;

    private final String type;

    private AutomatedAbrRule(BuilderImpl builder) {
        this.allowedRenditions = builder.allowedRenditions;
        this.forceIncludeRenditions = builder.forceIncludeRenditions;
        this.minBottomRenditionSize = builder.minBottomRenditionSize;
        this.minTopRenditionSize = builder.minTopRenditionSize;
        this.type = builder.type;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedRenditions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedRenditions() {
        return allowedRenditions != null && !(allowedRenditions instanceof SdkAutoConstructList);
    }

    /**
     * When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat one
     * rendition to allowedRenditions list
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedRenditions} method.
     * </p>
     * 
     * @return When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat one
     *         rendition to allowedRenditions list
     */
    public final List<AllowedRenditionSize> allowedRenditions() {
        return allowedRenditions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ForceIncludeRenditions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasForceIncludeRenditions() {
        return forceIncludeRenditions != null && !(forceIncludeRenditions instanceof SdkAutoConstructList);
    }

    /**
     * When customer adds the force include renditions rule for auto ABR ladder, they are required to add at leat one
     * rendition to forceIncludeRenditions list
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasForceIncludeRenditions} method.
     * </p>
     * 
     * @return When customer adds the force include renditions rule for auto ABR ladder, they are required to add at
     *         leat one rendition to forceIncludeRenditions list
     */
    public final List<ForceIncludeRenditionSize> forceIncludeRenditions() {
        return forceIncludeRenditions;
    }

    /**
     * Use Min bottom rendition size to specify a minimum size for the lowest resolution in your ABR stack. * The lowest
     * resolution in your ABR stack will be equal to or greater than the value that you enter. For example: If you
     * specify 640x360 the lowest resolution in your ABR stack will be equal to or greater than to 640x360. * If you
     * specify a Min top rendition size rule, the value that you specify for Min bottom rendition size must be less
     * than, or equal to, Min top rendition size.
     * 
     * @return Use Min bottom rendition size to specify a minimum size for the lowest resolution in your ABR stack. *
     *         The lowest resolution in your ABR stack will be equal to or greater than the value that you enter. For
     *         example: If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater than
     *         to 640x360. * If you specify a Min top rendition size rule, the value that you specify for Min bottom
     *         rendition size must be less than, or equal to, Min top rendition size.
     */
    public final MinBottomRenditionSize minBottomRenditionSize() {
        return minBottomRenditionSize;
    }

    /**
     * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The highest
     * resolution in your ABR stack will be equal to or greater than the value that you enter. For example: If you
     * specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. * If you
     * specify a value for Max resolution, the value that you specify for Min top rendition size must be less than, or
     * equal to, Max resolution.
     * 
     * @return Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
     *         highest resolution in your ABR stack will be equal to or greater than the value that you enter. For
     *         example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater
     *         than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min top
     *         rendition size must be less than, or equal to, Max resolution.
     */
    public final MinTopRenditionSize minTopRenditionSize() {
        return minTopRenditionSize;
    }

    /**
     * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The highest
     * resolution in your ABR stack will be equal to or greater than the value that you enter. For example: If you
     * specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. * If you
     * specify a value for Max resolution, the value that you specify for Min top rendition size must be less than, or
     * equal to, Max resolution. Use Min bottom rendition size to specify a minimum size for the lowest resolution in
     * your ABR stack. * The lowest resolution in your ABR stack will be equal to or greater than the value that you
     * enter. For example: If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater
     * than to 640x360. * If you specify a Min top rendition size rule, the value that you specify for Min bottom
     * rendition size must be less than, or equal to, Min top rendition size. Use Force include renditions to specify
     * one or more resolutions to include your ABR stack. * (Recommended) To optimize automated ABR, specify as few
     * resolutions as possible. * (Required) The number of resolutions that you specify must be equal to, or less than,
     * the Max renditions setting. * If you specify a Min top rendition size rule, specify at least one resolution that
     * is equal to, or greater than, Min top rendition size. * If you specify a Min bottom rendition size rule, only
     * specify resolutions that are equal to, or greater than, Min bottom rendition size. * If you specify a Force
     * include renditions rule, do not specify a separate rule for Allowed renditions. * Note: The ABR stack may include
     * other resolutions that you do not specify here, depending on the Max renditions setting. Use Allowed renditions
     * to specify a list of possible resolutions in your ABR stack. * (Required) The number of resolutions that you
     * specify must be equal to, or greater than, the Max renditions setting. * MediaConvert will create an ABR stack
     * exclusively from the list of resolutions that you specify. * Some resolutions in the Allowed renditions list may
     * not be included, however you can force a resolution to be included by setting Required to ENABLED. * You must
     * specify at least one resolution that is greater than or equal to any resolutions that you specify in Min top
     * rendition size or Min bottom rendition size. * If you specify Allowed renditions, you must not specify a separate
     * rule for Force include renditions.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
     *         highest resolution in your ABR stack will be equal to or greater than the value that you enter. For
     *         example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater
     *         than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min top
     *         rendition size must be less than, or equal to, Max resolution. Use Min bottom rendition size to specify a
     *         minimum size for the lowest resolution in your ABR stack. * The lowest resolution in your ABR stack will
     *         be equal to or greater than the value that you enter. For example: If you specify 640x360 the lowest
     *         resolution in your ABR stack will be equal to or greater than to 640x360. * If you specify a Min top
     *         rendition size rule, the value that you specify for Min bottom rendition size must be less than, or equal
     *         to, Min top rendition size. Use Force include renditions to specify one or more resolutions to include
     *         your ABR stack. * (Recommended) To optimize automated ABR, specify as few resolutions as possible. *
     *         (Required) The number of resolutions that you specify must be equal to, or less than, the Max renditions
     *         setting. * If you specify a Min top rendition size rule, specify at least one resolution that is equal
     *         to, or greater than, Min top rendition size. * If you specify a Min bottom rendition size rule, only
     *         specify resolutions that are equal to, or greater than, Min bottom rendition size. * If you specify a
     *         Force include renditions rule, do not specify a separate rule for Allowed renditions. * Note: The ABR
     *         stack may include other resolutions that you do not specify here, depending on the Max renditions
     *         setting. Use Allowed renditions to specify a list of possible resolutions in your ABR stack. * (Required)
     *         The number of resolutions that you specify must be equal to, or greater than, the Max renditions setting.
     *         * MediaConvert will create an ABR stack exclusively from the list of resolutions that you specify. * Some
     *         resolutions in the Allowed renditions list may not be included, however you can force a resolution to be
     *         included by setting Required to ENABLED. * You must specify at least one resolution that is greater than
     *         or equal to any resolutions that you specify in Min top rendition size or Min bottom rendition size. * If
     *         you specify Allowed renditions, you must not specify a separate rule for Force include renditions.
     * @see RuleType
     */
    public final RuleType type() {
        return RuleType.fromValue(type);
    }

    /**
     * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The highest
     * resolution in your ABR stack will be equal to or greater than the value that you enter. For example: If you
     * specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. * If you
     * specify a value for Max resolution, the value that you specify for Min top rendition size must be less than, or
     * equal to, Max resolution. Use Min bottom rendition size to specify a minimum size for the lowest resolution in
     * your ABR stack. * The lowest resolution in your ABR stack will be equal to or greater than the value that you
     * enter. For example: If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater
     * than to 640x360. * If you specify a Min top rendition size rule, the value that you specify for Min bottom
     * rendition size must be less than, or equal to, Min top rendition size. Use Force include renditions to specify
     * one or more resolutions to include your ABR stack. * (Recommended) To optimize automated ABR, specify as few
     * resolutions as possible. * (Required) The number of resolutions that you specify must be equal to, or less than,
     * the Max renditions setting. * If you specify a Min top rendition size rule, specify at least one resolution that
     * is equal to, or greater than, Min top rendition size. * If you specify a Min bottom rendition size rule, only
     * specify resolutions that are equal to, or greater than, Min bottom rendition size. * If you specify a Force
     * include renditions rule, do not specify a separate rule for Allowed renditions. * Note: The ABR stack may include
     * other resolutions that you do not specify here, depending on the Max renditions setting. Use Allowed renditions
     * to specify a list of possible resolutions in your ABR stack. * (Required) The number of resolutions that you
     * specify must be equal to, or greater than, the Max renditions setting. * MediaConvert will create an ABR stack
     * exclusively from the list of resolutions that you specify. * Some resolutions in the Allowed renditions list may
     * not be included, however you can force a resolution to be included by setting Required to ENABLED. * You must
     * specify at least one resolution that is greater than or equal to any resolutions that you specify in Min top
     * rendition size or Min bottom rendition size. * If you specify Allowed renditions, you must not specify a separate
     * rule for Force include renditions.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
     *         highest resolution in your ABR stack will be equal to or greater than the value that you enter. For
     *         example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater
     *         than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min top
     *         rendition size must be less than, or equal to, Max resolution. Use Min bottom rendition size to specify a
     *         minimum size for the lowest resolution in your ABR stack. * The lowest resolution in your ABR stack will
     *         be equal to or greater than the value that you enter. For example: If you specify 640x360 the lowest
     *         resolution in your ABR stack will be equal to or greater than to 640x360. * If you specify a Min top
     *         rendition size rule, the value that you specify for Min bottom rendition size must be less than, or equal
     *         to, Min top rendition size. Use Force include renditions to specify one or more resolutions to include
     *         your ABR stack. * (Recommended) To optimize automated ABR, specify as few resolutions as possible. *
     *         (Required) The number of resolutions that you specify must be equal to, or less than, the Max renditions
     *         setting. * If you specify a Min top rendition size rule, specify at least one resolution that is equal
     *         to, or greater than, Min top rendition size. * If you specify a Min bottom rendition size rule, only
     *         specify resolutions that are equal to, or greater than, Min bottom rendition size. * If you specify a
     *         Force include renditions rule, do not specify a separate rule for Allowed renditions. * Note: The ABR
     *         stack may include other resolutions that you do not specify here, depending on the Max renditions
     *         setting. Use Allowed renditions to specify a list of possible resolutions in your ABR stack. * (Required)
     *         The number of resolutions that you specify must be equal to, or greater than, the Max renditions setting.
     *         * MediaConvert will create an ABR stack exclusively from the list of resolutions that you specify. * Some
     *         resolutions in the Allowed renditions list may not be included, however you can force a resolution to be
     *         included by setting Required to ENABLED. * You must specify at least one resolution that is greater than
     *         or equal to any resolutions that you specify in Min top rendition size or Min bottom rendition size. * If
     *         you specify Allowed renditions, you must not specify a separate rule for Force include renditions.
     * @see RuleType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedRenditions() ? allowedRenditions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasForceIncludeRenditions() ? forceIncludeRenditions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(minBottomRenditionSize());
        hashCode = 31 * hashCode + Objects.hashCode(minTopRenditionSize());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutomatedAbrRule)) {
            return false;
        }
        AutomatedAbrRule other = (AutomatedAbrRule) obj;
        return hasAllowedRenditions() == other.hasAllowedRenditions()
                && Objects.equals(allowedRenditions(), other.allowedRenditions())
                && hasForceIncludeRenditions() == other.hasForceIncludeRenditions()
                && Objects.equals(forceIncludeRenditions(), other.forceIncludeRenditions())
                && Objects.equals(minBottomRenditionSize(), other.minBottomRenditionSize())
                && Objects.equals(minTopRenditionSize(), other.minTopRenditionSize())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutomatedAbrRule").add("AllowedRenditions", hasAllowedRenditions() ? allowedRenditions() : null)
                .add("ForceIncludeRenditions", hasForceIncludeRenditions() ? forceIncludeRenditions() : null)
                .add("MinBottomRenditionSize", minBottomRenditionSize()).add("MinTopRenditionSize", minTopRenditionSize())
                .add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllowedRenditions":
            return Optional.ofNullable(clazz.cast(allowedRenditions()));
        case "ForceIncludeRenditions":
            return Optional.ofNullable(clazz.cast(forceIncludeRenditions()));
        case "MinBottomRenditionSize":
            return Optional.ofNullable(clazz.cast(minBottomRenditionSize()));
        case "MinTopRenditionSize":
            return Optional.ofNullable(clazz.cast(minTopRenditionSize()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutomatedAbrRule, T> g) {
        return obj -> g.apply((AutomatedAbrRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutomatedAbrRule> {
        /**
         * When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat one
         * rendition to allowedRenditions list
         * 
         * @param allowedRenditions
         *        When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat
         *        one rendition to allowedRenditions list
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedRenditions(Collection<AllowedRenditionSize> allowedRenditions);

        /**
         * When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat one
         * rendition to allowedRenditions list
         * 
         * @param allowedRenditions
         *        When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat
         *        one rendition to allowedRenditions list
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedRenditions(AllowedRenditionSize... allowedRenditions);

        /**
         * When customer adds the allowed renditions rule for auto ABR ladder, they are required to add at leat one
         * rendition to allowedRenditions list This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconvert.model.AllowedRenditionSize.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.mediaconvert.model.AllowedRenditionSize#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconvert.model.AllowedRenditionSize.Builder#build()} is called
         * immediately and its result is passed to {@link #allowedRenditions(List<AllowedRenditionSize>)}.
         * 
         * @param allowedRenditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconvert.model.AllowedRenditionSize.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allowedRenditions(java.util.Collection<AllowedRenditionSize>)
         */
        Builder allowedRenditions(Consumer<AllowedRenditionSize.Builder>... allowedRenditions);

        /**
         * When customer adds the force include renditions rule for auto ABR ladder, they are required to add at leat
         * one rendition to forceIncludeRenditions list
         * 
         * @param forceIncludeRenditions
         *        When customer adds the force include renditions rule for auto ABR ladder, they are required to add at
         *        leat one rendition to forceIncludeRenditions list
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceIncludeRenditions(Collection<ForceIncludeRenditionSize> forceIncludeRenditions);

        /**
         * When customer adds the force include renditions rule for auto ABR ladder, they are required to add at leat
         * one rendition to forceIncludeRenditions list
         * 
         * @param forceIncludeRenditions
         *        When customer adds the force include renditions rule for auto ABR ladder, they are required to add at
         *        leat one rendition to forceIncludeRenditions list
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceIncludeRenditions(ForceIncludeRenditionSize... forceIncludeRenditions);

        /**
         * When customer adds the force include renditions rule for auto ABR ladder, they are required to add at leat
         * one rendition to forceIncludeRenditions list This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconvert.model.ForceIncludeRenditionSize.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.mediaconvert.model.ForceIncludeRenditionSize#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconvert.model.ForceIncludeRenditionSize.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #forceIncludeRenditions(List<ForceIncludeRenditionSize>)}.
         * 
         * @param forceIncludeRenditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconvert.model.ForceIncludeRenditionSize.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forceIncludeRenditions(java.util.Collection<ForceIncludeRenditionSize>)
         */
        Builder forceIncludeRenditions(Consumer<ForceIncludeRenditionSize.Builder>... forceIncludeRenditions);

        /**
         * Use Min bottom rendition size to specify a minimum size for the lowest resolution in your ABR stack. * The
         * lowest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater than to 640x360. *
         * If you specify a Min top rendition size rule, the value that you specify for Min bottom rendition size must
         * be less than, or equal to, Min top rendition size.
         * 
         * @param minBottomRenditionSize
         *        Use Min bottom rendition size to specify a minimum size for the lowest resolution in your ABR stack. *
         *        The lowest resolution in your ABR stack will be equal to or greater than the value that you enter. For
         *        example: If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater
         *        than to 640x360. * If you specify a Min top rendition size rule, the value that you specify for Min
         *        bottom rendition size must be less than, or equal to, Min top rendition size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minBottomRenditionSize(MinBottomRenditionSize minBottomRenditionSize);

        /**
         * Use Min bottom rendition size to specify a minimum size for the lowest resolution in your ABR stack. * The
         * lowest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 640x360 the lowest resolution in your ABR stack will be equal to or greater than to 640x360. *
         * If you specify a Min top rendition size rule, the value that you specify for Min bottom rendition size must
         * be less than, or equal to, Min top rendition size. This is a convenience method that creates an instance of
         * the {@link MinBottomRenditionSize.Builder} avoiding the need to create one manually via
         * {@link MinBottomRenditionSize#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MinBottomRenditionSize.Builder#build()} is called immediately and
         * its result is passed to {@link #minBottomRenditionSize(MinBottomRenditionSize)}.
         * 
         * @param minBottomRenditionSize
         *        a consumer that will call methods on {@link MinBottomRenditionSize.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #minBottomRenditionSize(MinBottomRenditionSize)
         */
        default Builder minBottomRenditionSize(Consumer<MinBottomRenditionSize.Builder> minBottomRenditionSize) {
            return minBottomRenditionSize(MinBottomRenditionSize.builder().applyMutation(minBottomRenditionSize).build());
        }

        /**
         * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
         * highest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. *
         * If you specify a value for Max resolution, the value that you specify for Min top rendition size must be less
         * than, or equal to, Max resolution.
         * 
         * @param minTopRenditionSize
         *        Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. *
         *        The highest resolution in your ABR stack will be equal to or greater than the value that you enter.
         *        For example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or
         *        greater than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min
         *        top rendition size must be less than, or equal to, Max resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTopRenditionSize(MinTopRenditionSize minTopRenditionSize);

        /**
         * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
         * highest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. *
         * If you specify a value for Max resolution, the value that you specify for Min top rendition size must be less
         * than, or equal to, Max resolution. This is a convenience method that creates an instance of the
         * {@link MinTopRenditionSize.Builder} avoiding the need to create one manually via
         * {@link MinTopRenditionSize#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MinTopRenditionSize.Builder#build()} is called immediately and
         * its result is passed to {@link #minTopRenditionSize(MinTopRenditionSize)}.
         * 
         * @param minTopRenditionSize
         *        a consumer that will call methods on {@link MinTopRenditionSize.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #minTopRenditionSize(MinTopRenditionSize)
         */
        default Builder minTopRenditionSize(Consumer<MinTopRenditionSize.Builder> minTopRenditionSize) {
            return minTopRenditionSize(MinTopRenditionSize.builder().applyMutation(minTopRenditionSize).build());
        }

        /**
         * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
         * highest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. *
         * If you specify a value for Max resolution, the value that you specify for Min top rendition size must be less
         * than, or equal to, Max resolution. Use Min bottom rendition size to specify a minimum size for the lowest
         * resolution in your ABR stack. * The lowest resolution in your ABR stack will be equal to or greater than the
         * value that you enter. For example: If you specify 640x360 the lowest resolution in your ABR stack will be
         * equal to or greater than to 640x360. * If you specify a Min top rendition size rule, the value that you
         * specify for Min bottom rendition size must be less than, or equal to, Min top rendition size. Use Force
         * include renditions to specify one or more resolutions to include your ABR stack. * (Recommended) To optimize
         * automated ABR, specify as few resolutions as possible. * (Required) The number of resolutions that you
         * specify must be equal to, or less than, the Max renditions setting. * If you specify a Min top rendition size
         * rule, specify at least one resolution that is equal to, or greater than, Min top rendition size. * If you
         * specify a Min bottom rendition size rule, only specify resolutions that are equal to, or greater than, Min
         * bottom rendition size. * If you specify a Force include renditions rule, do not specify a separate rule for
         * Allowed renditions. * Note: The ABR stack may include other resolutions that you do not specify here,
         * depending on the Max renditions setting. Use Allowed renditions to specify a list of possible resolutions in
         * your ABR stack. * (Required) The number of resolutions that you specify must be equal to, or greater than,
         * the Max renditions setting. * MediaConvert will create an ABR stack exclusively from the list of resolutions
         * that you specify. * Some resolutions in the Allowed renditions list may not be included, however you can
         * force a resolution to be included by setting Required to ENABLED. * You must specify at least one resolution
         * that is greater than or equal to any resolutions that you specify in Min top rendition size or Min bottom
         * rendition size. * If you specify Allowed renditions, you must not specify a separate rule for Force include
         * renditions.
         * 
         * @param type
         *        Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. *
         *        The highest resolution in your ABR stack will be equal to or greater than the value that you enter.
         *        For example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or
         *        greater than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min
         *        top rendition size must be less than, or equal to, Max resolution. Use Min bottom rendition size to
         *        specify a minimum size for the lowest resolution in your ABR stack. * The lowest resolution in your
         *        ABR stack will be equal to or greater than the value that you enter. For example: If you specify
         *        640x360 the lowest resolution in your ABR stack will be equal to or greater than to 640x360. * If you
         *        specify a Min top rendition size rule, the value that you specify for Min bottom rendition size must
         *        be less than, or equal to, Min top rendition size. Use Force include renditions to specify one or more
         *        resolutions to include your ABR stack. * (Recommended) To optimize automated ABR, specify as few
         *        resolutions as possible. * (Required) The number of resolutions that you specify must be equal to, or
         *        less than, the Max renditions setting. * If you specify a Min top rendition size rule, specify at
         *        least one resolution that is equal to, or greater than, Min top rendition size. * If you specify a Min
         *        bottom rendition size rule, only specify resolutions that are equal to, or greater than, Min bottom
         *        rendition size. * If you specify a Force include renditions rule, do not specify a separate rule for
         *        Allowed renditions. * Note: The ABR stack may include other resolutions that you do not specify here,
         *        depending on the Max renditions setting. Use Allowed renditions to specify a list of possible
         *        resolutions in your ABR stack. * (Required) The number of resolutions that you specify must be equal
         *        to, or greater than, the Max renditions setting. * MediaConvert will create an ABR stack exclusively
         *        from the list of resolutions that you specify. * Some resolutions in the Allowed renditions list may
         *        not be included, however you can force a resolution to be included by setting Required to ENABLED. *
         *        You must specify at least one resolution that is greater than or equal to any resolutions that you
         *        specify in Min top rendition size or Min bottom rendition size. * If you specify Allowed renditions,
         *        you must not specify a separate rule for Force include renditions.
         * @see RuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleType
         */
        Builder type(String type);

        /**
         * Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. * The
         * highest resolution in your ABR stack will be equal to or greater than the value that you enter. For example:
         * If you specify 1280x720 the highest resolution in your ABR stack will be equal to or greater than 1280x720. *
         * If you specify a value for Max resolution, the value that you specify for Min top rendition size must be less
         * than, or equal to, Max resolution. Use Min bottom rendition size to specify a minimum size for the lowest
         * resolution in your ABR stack. * The lowest resolution in your ABR stack will be equal to or greater than the
         * value that you enter. For example: If you specify 640x360 the lowest resolution in your ABR stack will be
         * equal to or greater than to 640x360. * If you specify a Min top rendition size rule, the value that you
         * specify for Min bottom rendition size must be less than, or equal to, Min top rendition size. Use Force
         * include renditions to specify one or more resolutions to include your ABR stack. * (Recommended) To optimize
         * automated ABR, specify as few resolutions as possible. * (Required) The number of resolutions that you
         * specify must be equal to, or less than, the Max renditions setting. * If you specify a Min top rendition size
         * rule, specify at least one resolution that is equal to, or greater than, Min top rendition size. * If you
         * specify a Min bottom rendition size rule, only specify resolutions that are equal to, or greater than, Min
         * bottom rendition size. * If you specify a Force include renditions rule, do not specify a separate rule for
         * Allowed renditions. * Note: The ABR stack may include other resolutions that you do not specify here,
         * depending on the Max renditions setting. Use Allowed renditions to specify a list of possible resolutions in
         * your ABR stack. * (Required) The number of resolutions that you specify must be equal to, or greater than,
         * the Max renditions setting. * MediaConvert will create an ABR stack exclusively from the list of resolutions
         * that you specify. * Some resolutions in the Allowed renditions list may not be included, however you can
         * force a resolution to be included by setting Required to ENABLED. * You must specify at least one resolution
         * that is greater than or equal to any resolutions that you specify in Min top rendition size or Min bottom
         * rendition size. * If you specify Allowed renditions, you must not specify a separate rule for Force include
         * renditions.
         * 
         * @param type
         *        Use Min top rendition size to specify a minimum size for the highest resolution in your ABR stack. *
         *        The highest resolution in your ABR stack will be equal to or greater than the value that you enter.
         *        For example: If you specify 1280x720 the highest resolution in your ABR stack will be equal to or
         *        greater than 1280x720. * If you specify a value for Max resolution, the value that you specify for Min
         *        top rendition size must be less than, or equal to, Max resolution. Use Min bottom rendition size to
         *        specify a minimum size for the lowest resolution in your ABR stack. * The lowest resolution in your
         *        ABR stack will be equal to or greater than the value that you enter. For example: If you specify
         *        640x360 the lowest resolution in your ABR stack will be equal to or greater than to 640x360. * If you
         *        specify a Min top rendition size rule, the value that you specify for Min bottom rendition size must
         *        be less than, or equal to, Min top rendition size. Use Force include renditions to specify one or more
         *        resolutions to include your ABR stack. * (Recommended) To optimize automated ABR, specify as few
         *        resolutions as possible. * (Required) The number of resolutions that you specify must be equal to, or
         *        less than, the Max renditions setting. * If you specify a Min top rendition size rule, specify at
         *        least one resolution that is equal to, or greater than, Min top rendition size. * If you specify a Min
         *        bottom rendition size rule, only specify resolutions that are equal to, or greater than, Min bottom
         *        rendition size. * If you specify a Force include renditions rule, do not specify a separate rule for
         *        Allowed renditions. * Note: The ABR stack may include other resolutions that you do not specify here,
         *        depending on the Max renditions setting. Use Allowed renditions to specify a list of possible
         *        resolutions in your ABR stack. * (Required) The number of resolutions that you specify must be equal
         *        to, or greater than, the Max renditions setting. * MediaConvert will create an ABR stack exclusively
         *        from the list of resolutions that you specify. * Some resolutions in the Allowed renditions list may
         *        not be included, however you can force a resolution to be included by setting Required to ENABLED. *
         *        You must specify at least one resolution that is greater than or equal to any resolutions that you
         *        specify in Min top rendition size or Min bottom rendition size. * If you specify Allowed renditions,
         *        you must not specify a separate rule for Force include renditions.
         * @see RuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleType
         */
        Builder type(RuleType type);
    }

    static final class BuilderImpl implements Builder {
        private List<AllowedRenditionSize> allowedRenditions = DefaultSdkAutoConstructList.getInstance();

        private List<ForceIncludeRenditionSize> forceIncludeRenditions = DefaultSdkAutoConstructList.getInstance();

        private MinBottomRenditionSize minBottomRenditionSize;

        private MinTopRenditionSize minTopRenditionSize;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(AutomatedAbrRule model) {
            allowedRenditions(model.allowedRenditions);
            forceIncludeRenditions(model.forceIncludeRenditions);
            minBottomRenditionSize(model.minBottomRenditionSize);
            minTopRenditionSize(model.minTopRenditionSize);
            type(model.type);
        }

        public final List<AllowedRenditionSize.Builder> getAllowedRenditions() {
            List<AllowedRenditionSize.Builder> result = ___listOfAllowedRenditionSizeCopier.copyToBuilder(this.allowedRenditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAllowedRenditions(Collection<AllowedRenditionSize.BuilderImpl> allowedRenditions) {
            this.allowedRenditions = ___listOfAllowedRenditionSizeCopier.copyFromBuilder(allowedRenditions);
        }

        @Override
        public final Builder allowedRenditions(Collection<AllowedRenditionSize> allowedRenditions) {
            this.allowedRenditions = ___listOfAllowedRenditionSizeCopier.copy(allowedRenditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedRenditions(AllowedRenditionSize... allowedRenditions) {
            allowedRenditions(Arrays.asList(allowedRenditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedRenditions(Consumer<AllowedRenditionSize.Builder>... allowedRenditions) {
            allowedRenditions(Stream.of(allowedRenditions).map(c -> AllowedRenditionSize.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ForceIncludeRenditionSize.Builder> getForceIncludeRenditions() {
            List<ForceIncludeRenditionSize.Builder> result = ___listOfForceIncludeRenditionSizeCopier
                    .copyToBuilder(this.forceIncludeRenditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setForceIncludeRenditions(Collection<ForceIncludeRenditionSize.BuilderImpl> forceIncludeRenditions) {
            this.forceIncludeRenditions = ___listOfForceIncludeRenditionSizeCopier.copyFromBuilder(forceIncludeRenditions);
        }

        @Override
        public final Builder forceIncludeRenditions(Collection<ForceIncludeRenditionSize> forceIncludeRenditions) {
            this.forceIncludeRenditions = ___listOfForceIncludeRenditionSizeCopier.copy(forceIncludeRenditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forceIncludeRenditions(ForceIncludeRenditionSize... forceIncludeRenditions) {
            forceIncludeRenditions(Arrays.asList(forceIncludeRenditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forceIncludeRenditions(Consumer<ForceIncludeRenditionSize.Builder>... forceIncludeRenditions) {
            forceIncludeRenditions(Stream.of(forceIncludeRenditions)
                    .map(c -> ForceIncludeRenditionSize.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final MinBottomRenditionSize.Builder getMinBottomRenditionSize() {
            return minBottomRenditionSize != null ? minBottomRenditionSize.toBuilder() : null;
        }

        public final void setMinBottomRenditionSize(MinBottomRenditionSize.BuilderImpl minBottomRenditionSize) {
            this.minBottomRenditionSize = minBottomRenditionSize != null ? minBottomRenditionSize.build() : null;
        }

        @Override
        public final Builder minBottomRenditionSize(MinBottomRenditionSize minBottomRenditionSize) {
            this.minBottomRenditionSize = minBottomRenditionSize;
            return this;
        }

        public final MinTopRenditionSize.Builder getMinTopRenditionSize() {
            return minTopRenditionSize != null ? minTopRenditionSize.toBuilder() : null;
        }

        public final void setMinTopRenditionSize(MinTopRenditionSize.BuilderImpl minTopRenditionSize) {
            this.minTopRenditionSize = minTopRenditionSize != null ? minTopRenditionSize.build() : null;
        }

        @Override
        public final Builder minTopRenditionSize(MinTopRenditionSize minTopRenditionSize) {
            this.minTopRenditionSize = minTopRenditionSize;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(RuleType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public AutomatedAbrRule build() {
            return new AutomatedAbrRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
