/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of a snapshot or cluster export to Amazon S3.
 * </p>
 * <p>
 * This data type is used as a response element in the <code>DescribeExportTasks</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExportTask implements SdkPojo, Serializable, ToCopyableBuilder<ExportTask.Builder, ExportTask> {
    private static final SdkField<String> EXPORT_TASK_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExportTaskIdentifier").getter(getter(ExportTask::exportTaskIdentifier))
            .setter(setter(Builder::exportTaskIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportTaskIdentifier").build())
            .build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(ExportTask::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<List<String>> EXPORT_ONLY_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExportOnly")
            .getter(getter(ExportTask::exportOnly))
            .setter(setter(Builder::exportOnly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportOnly").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> SNAPSHOT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SnapshotTime").getter(getter(ExportTask::snapshotTime)).setter(setter(Builder::snapshotTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotTime").build()).build();

    private static final SdkField<Instant> TASK_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TaskStartTime").getter(getter(ExportTask::taskStartTime)).setter(setter(Builder::taskStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskStartTime").build()).build();

    private static final SdkField<Instant> TASK_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TaskEndTime").getter(getter(ExportTask::taskEndTime)).setter(setter(Builder::taskEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskEndTime").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Bucket").getter(getter(ExportTask::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Bucket").build()).build();

    private static final SdkField<String> S3_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Prefix").getter(getter(ExportTask::s3Prefix)).setter(setter(Builder::s3Prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Prefix").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(ExportTask::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(ExportTask::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ExportTask::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> PERCENT_PROGRESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("PercentProgress").getter(getter(ExportTask::percentProgress)).setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentProgress").build()).build();

    private static final SdkField<Integer> TOTAL_EXTRACTED_DATA_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalExtractedDataInGB").getter(getter(ExportTask::totalExtractedDataInGB))
            .setter(setter(Builder::totalExtractedDataInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalExtractedDataInGB").build())
            .build();

    private static final SdkField<String> FAILURE_CAUSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureCause").getter(getter(ExportTask::failureCause)).setter(setter(Builder::failureCause))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureCause").build()).build();

    private static final SdkField<String> WARNING_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WarningMessage").getter(getter(ExportTask::warningMessage)).setter(setter(Builder::warningMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarningMessage").build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(ExportTask::sourceTypeAsString)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPORT_TASK_IDENTIFIER_FIELD,
            SOURCE_ARN_FIELD, EXPORT_ONLY_FIELD, SNAPSHOT_TIME_FIELD, TASK_START_TIME_FIELD, TASK_END_TIME_FIELD,
            S3_BUCKET_FIELD, S3_PREFIX_FIELD, IAM_ROLE_ARN_FIELD, KMS_KEY_ID_FIELD, STATUS_FIELD, PERCENT_PROGRESS_FIELD,
            TOTAL_EXTRACTED_DATA_IN_GB_FIELD, FAILURE_CAUSE_FIELD, WARNING_MESSAGE_FIELD, SOURCE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String exportTaskIdentifier;

    private final String sourceArn;

    private final List<String> exportOnly;

    private final Instant snapshotTime;

    private final Instant taskStartTime;

    private final Instant taskEndTime;

    private final String s3Bucket;

    private final String s3Prefix;

    private final String iamRoleArn;

    private final String kmsKeyId;

    private final String status;

    private final Integer percentProgress;

    private final Integer totalExtractedDataInGB;

    private final String failureCause;

    private final String warningMessage;

    private final String sourceType;

    private ExportTask(BuilderImpl builder) {
        this.exportTaskIdentifier = builder.exportTaskIdentifier;
        this.sourceArn = builder.sourceArn;
        this.exportOnly = builder.exportOnly;
        this.snapshotTime = builder.snapshotTime;
        this.taskStartTime = builder.taskStartTime;
        this.taskEndTime = builder.taskEndTime;
        this.s3Bucket = builder.s3Bucket;
        this.s3Prefix = builder.s3Prefix;
        this.iamRoleArn = builder.iamRoleArn;
        this.kmsKeyId = builder.kmsKeyId;
        this.status = builder.status;
        this.percentProgress = builder.percentProgress;
        this.totalExtractedDataInGB = builder.totalExtractedDataInGB;
        this.failureCause = builder.failureCause;
        this.warningMessage = builder.warningMessage;
        this.sourceType = builder.sourceType;
    }

    /**
     * <p>
     * A unique identifier for the snapshot or cluster export task. This ID isn't an identifier for the Amazon S3 bucket
     * where the data is exported.
     * </p>
     * 
     * @return A unique identifier for the snapshot or cluster export task. This ID isn't an identifier for the Amazon
     *         S3 bucket where the data is exported.
     */
    public final String exportTaskIdentifier() {
        return exportTaskIdentifier;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the snapshot or cluster exported to Amazon S3.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the snapshot or cluster exported to Amazon S3.
     */
    public final String sourceArn() {
        return sourceArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExportOnly property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExportOnly() {
        return exportOnly != null && !(exportOnly instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The data exported from the snapshot or cluster.
     * </p>
     * <p>
     * Valid Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>database</code> - Export all the data from a specified database.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format is valid
     * only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster. This
     * format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This format is
     * valid only for RDS for PostgreSQL and Aurora PostgreSQL.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExportOnly} method.
     * </p>
     * 
     * @return The data exported from the snapshot or cluster.</p>
     *         <p>
     *         Valid Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>database</code> - Export all the data from a specified database.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format is
     *         valid only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster.
     *         This format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This format
     *         is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
     *         </p>
     *         </li>
     */
    public final List<String> exportOnly() {
        return exportOnly;
    }

    /**
     * <p>
     * The time when the snapshot was created.
     * </p>
     * 
     * @return The time when the snapshot was created.
     */
    public final Instant snapshotTime() {
        return snapshotTime;
    }

    /**
     * <p>
     * The time when the snapshot or cluster export task started.
     * </p>
     * 
     * @return The time when the snapshot or cluster export task started.
     */
    public final Instant taskStartTime() {
        return taskStartTime;
    }

    /**
     * <p>
     * The time when the snapshot or cluster export task ended.
     * </p>
     * 
     * @return The time when the snapshot or cluster export task ended.
     */
    public final Instant taskEndTime() {
        return taskEndTime;
    }

    /**
     * <p>
     * The Amazon S3 bucket where the snapshot or cluster is exported to.
     * </p>
     * 
     * @return The Amazon S3 bucket where the snapshot or cluster is exported to.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    /**
     * <p>
     * The Amazon S3 bucket prefix that is the file name and path of the exported data.
     * </p>
     * 
     * @return The Amazon S3 bucket prefix that is the file name and path of the exported data.
     */
    public final String s3Prefix() {
        return s3Prefix;
    }

    /**
     * <p>
     * The name of the IAM role that is used to write to Amazon S3 when exporting a snapshot or cluster.
     * </p>
     * 
     * @return The name of the IAM role that is used to write to Amazon S3 when exporting a snapshot or cluster.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * The key identifier of the Amazon Web Services KMS key that is used to encrypt the data when it's exported to
     * Amazon S3. The KMS key identifier is its key ARN, key ID, alias ARN, or alias name. The IAM role used for the
     * export must have encryption and decryption permissions to use this KMS key.
     * </p>
     * 
     * @return The key identifier of the Amazon Web Services KMS key that is used to encrypt the data when it's exported
     *         to Amazon S3. The KMS key identifier is its key ARN, key ID, alias ARN, or alias name. The IAM role used
     *         for the export must have encryption and decryption permissions to use this KMS key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The progress status of the export task. The status can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CANCELED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELING</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STARTING</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The progress status of the export task. The status can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CANCELED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELING</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STARTING</code>
     *         </p>
     *         </li>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The progress of the snapshot or cluster export task as a percentage.
     * </p>
     * 
     * @return The progress of the snapshot or cluster export task as a percentage.
     */
    public final Integer percentProgress() {
        return percentProgress;
    }

    /**
     * <p>
     * The total amount of data exported, in gigabytes.
     * </p>
     * 
     * @return The total amount of data exported, in gigabytes.
     */
    public final Integer totalExtractedDataInGB() {
        return totalExtractedDataInGB;
    }

    /**
     * <p>
     * The reason the export failed, if it failed.
     * </p>
     * 
     * @return The reason the export failed, if it failed.
     */
    public final String failureCause() {
        return failureCause;
    }

    /**
     * <p>
     * A warning about the snapshot or cluster export task.
     * </p>
     * 
     * @return A warning about the snapshot or cluster export task.
     */
    public final String warningMessage() {
        return warningMessage;
    }

    /**
     * <p>
     * The type of source for the export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link ExportSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The type of source for the export.
     * @see ExportSourceType
     */
    public final ExportSourceType sourceType() {
        return ExportSourceType.fromValue(sourceType);
    }

    /**
     * <p>
     * The type of source for the export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link ExportSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The type of source for the export.
     * @see ExportSourceType
     */
    public final String sourceTypeAsString() {
        return sourceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(exportTaskIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasExportOnly() ? exportOnly() : null);
        hashCode = 31 * hashCode + Objects.hashCode(snapshotTime());
        hashCode = 31 * hashCode + Objects.hashCode(taskStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(taskEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(s3Prefix());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(totalExtractedDataInGB());
        hashCode = 31 * hashCode + Objects.hashCode(failureCause());
        hashCode = 31 * hashCode + Objects.hashCode(warningMessage());
        hashCode = 31 * hashCode + Objects.hashCode(sourceTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExportTask)) {
            return false;
        }
        ExportTask other = (ExportTask) obj;
        return Objects.equals(exportTaskIdentifier(), other.exportTaskIdentifier())
                && Objects.equals(sourceArn(), other.sourceArn()) && hasExportOnly() == other.hasExportOnly()
                && Objects.equals(exportOnly(), other.exportOnly()) && Objects.equals(snapshotTime(), other.snapshotTime())
                && Objects.equals(taskStartTime(), other.taskStartTime()) && Objects.equals(taskEndTime(), other.taskEndTime())
                && Objects.equals(s3Bucket(), other.s3Bucket()) && Objects.equals(s3Prefix(), other.s3Prefix())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(status(), other.status()) && Objects.equals(percentProgress(), other.percentProgress())
                && Objects.equals(totalExtractedDataInGB(), other.totalExtractedDataInGB())
                && Objects.equals(failureCause(), other.failureCause())
                && Objects.equals(warningMessage(), other.warningMessage())
                && Objects.equals(sourceTypeAsString(), other.sourceTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExportTask").add("ExportTaskIdentifier", exportTaskIdentifier()).add("SourceArn", sourceArn())
                .add("ExportOnly", hasExportOnly() ? exportOnly() : null).add("SnapshotTime", snapshotTime())
                .add("TaskStartTime", taskStartTime()).add("TaskEndTime", taskEndTime()).add("S3Bucket", s3Bucket())
                .add("S3Prefix", s3Prefix()).add("IamRoleArn", iamRoleArn()).add("KmsKeyId", kmsKeyId()).add("Status", status())
                .add("PercentProgress", percentProgress()).add("TotalExtractedDataInGB", totalExtractedDataInGB())
                .add("FailureCause", failureCause()).add("WarningMessage", warningMessage())
                .add("SourceType", sourceTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExportTaskIdentifier":
            return Optional.ofNullable(clazz.cast(exportTaskIdentifier()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "ExportOnly":
            return Optional.ofNullable(clazz.cast(exportOnly()));
        case "SnapshotTime":
            return Optional.ofNullable(clazz.cast(snapshotTime()));
        case "TaskStartTime":
            return Optional.ofNullable(clazz.cast(taskStartTime()));
        case "TaskEndTime":
            return Optional.ofNullable(clazz.cast(taskEndTime()));
        case "S3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "S3Prefix":
            return Optional.ofNullable(clazz.cast(s3Prefix()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "PercentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        case "TotalExtractedDataInGB":
            return Optional.ofNullable(clazz.cast(totalExtractedDataInGB()));
        case "FailureCause":
            return Optional.ofNullable(clazz.cast(failureCause()));
        case "WarningMessage":
            return Optional.ofNullable(clazz.cast(warningMessage()));
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExportTask, T> g) {
        return obj -> g.apply((ExportTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExportTask> {
        /**
         * <p>
         * A unique identifier for the snapshot or cluster export task. This ID isn't an identifier for the Amazon S3
         * bucket where the data is exported.
         * </p>
         * 
         * @param exportTaskIdentifier
         *        A unique identifier for the snapshot or cluster export task. This ID isn't an identifier for the
         *        Amazon S3 bucket where the data is exported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportTaskIdentifier(String exportTaskIdentifier);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the snapshot or cluster exported to Amazon S3.
         * </p>
         * 
         * @param sourceArn
         *        The Amazon Resource Name (ARN) of the snapshot or cluster exported to Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * The data exported from the snapshot or cluster.
         * </p>
         * <p>
         * Valid Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>database</code> - Export all the data from a specified database.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format is
         * valid only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster. This
         * format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This format is
         * valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         * </p>
         * </li>
         * </ul>
         * 
         * @param exportOnly
         *        The data exported from the snapshot or cluster.</p>
         *        <p>
         *        Valid Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>database</code> - Export all the data from a specified database.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format
         *        is valid only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster.
         *        This format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This
         *        format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportOnly(Collection<String> exportOnly);

        /**
         * <p>
         * The data exported from the snapshot or cluster.
         * </p>
         * <p>
         * Valid Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>database</code> - Export all the data from a specified database.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format is
         * valid only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster. This
         * format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This format is
         * valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         * </p>
         * </li>
         * </ul>
         * 
         * @param exportOnly
         *        The data exported from the snapshot or cluster.</p>
         *        <p>
         *        Valid Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>database</code> - Export all the data from a specified database.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.table</code> <i>table-name</i> - Export a table of the snapshot or cluster. This format
         *        is valid only for RDS for MySQL, RDS for MariaDB, and Aurora MySQL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.schema</code> <i>schema-name</i> - Export a database schema of the snapshot or cluster.
         *        This format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>database.schema.table</code> <i>table-name</i> - Export a table of the database schema. This
         *        format is valid only for RDS for PostgreSQL and Aurora PostgreSQL.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportOnly(String... exportOnly);

        /**
         * <p>
         * The time when the snapshot was created.
         * </p>
         * 
         * @param snapshotTime
         *        The time when the snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotTime(Instant snapshotTime);

        /**
         * <p>
         * The time when the snapshot or cluster export task started.
         * </p>
         * 
         * @param taskStartTime
         *        The time when the snapshot or cluster export task started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskStartTime(Instant taskStartTime);

        /**
         * <p>
         * The time when the snapshot or cluster export task ended.
         * </p>
         * 
         * @param taskEndTime
         *        The time when the snapshot or cluster export task ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskEndTime(Instant taskEndTime);

        /**
         * <p>
         * The Amazon S3 bucket where the snapshot or cluster is exported to.
         * </p>
         * 
         * @param s3Bucket
         *        The Amazon S3 bucket where the snapshot or cluster is exported to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The Amazon S3 bucket prefix that is the file name and path of the exported data.
         * </p>
         * 
         * @param s3Prefix
         *        The Amazon S3 bucket prefix that is the file name and path of the exported data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Prefix(String s3Prefix);

        /**
         * <p>
         * The name of the IAM role that is used to write to Amazon S3 when exporting a snapshot or cluster.
         * </p>
         * 
         * @param iamRoleArn
         *        The name of the IAM role that is used to write to Amazon S3 when exporting a snapshot or cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * The key identifier of the Amazon Web Services KMS key that is used to encrypt the data when it's exported to
         * Amazon S3. The KMS key identifier is its key ARN, key ID, alias ARN, or alias name. The IAM role used for the
         * export must have encryption and decryption permissions to use this KMS key.
         * </p>
         * 
         * @param kmsKeyId
         *        The key identifier of the Amazon Web Services KMS key that is used to encrypt the data when it's
         *        exported to Amazon S3. The KMS key identifier is its key ARN, key ID, alias ARN, or alias name. The
         *        IAM role used for the export must have encryption and decryption permissions to use this KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The progress status of the export task. The status can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CANCELED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELING</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETE</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STARTING</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The progress status of the export task. The status can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CANCELED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELING</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETE</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STARTING</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The progress of the snapshot or cluster export task as a percentage.
         * </p>
         * 
         * @param percentProgress
         *        The progress of the snapshot or cluster export task as a percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(Integer percentProgress);

        /**
         * <p>
         * The total amount of data exported, in gigabytes.
         * </p>
         * 
         * @param totalExtractedDataInGB
         *        The total amount of data exported, in gigabytes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalExtractedDataInGB(Integer totalExtractedDataInGB);

        /**
         * <p>
         * The reason the export failed, if it failed.
         * </p>
         * 
         * @param failureCause
         *        The reason the export failed, if it failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureCause(String failureCause);

        /**
         * <p>
         * A warning about the snapshot or cluster export task.
         * </p>
         * 
         * @param warningMessage
         *        A warning about the snapshot or cluster export task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warningMessage(String warningMessage);

        /**
         * <p>
         * The type of source for the export.
         * </p>
         * 
         * @param sourceType
         *        The type of source for the export.
         * @see ExportSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExportSourceType
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * The type of source for the export.
         * </p>
         * 
         * @param sourceType
         *        The type of source for the export.
         * @see ExportSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExportSourceType
         */
        Builder sourceType(ExportSourceType sourceType);
    }

    static final class BuilderImpl implements Builder {
        private String exportTaskIdentifier;

        private String sourceArn;

        private List<String> exportOnly = DefaultSdkAutoConstructList.getInstance();

        private Instant snapshotTime;

        private Instant taskStartTime;

        private Instant taskEndTime;

        private String s3Bucket;

        private String s3Prefix;

        private String iamRoleArn;

        private String kmsKeyId;

        private String status;

        private Integer percentProgress;

        private Integer totalExtractedDataInGB;

        private String failureCause;

        private String warningMessage;

        private String sourceType;

        private BuilderImpl() {
        }

        private BuilderImpl(ExportTask model) {
            exportTaskIdentifier(model.exportTaskIdentifier);
            sourceArn(model.sourceArn);
            exportOnly(model.exportOnly);
            snapshotTime(model.snapshotTime);
            taskStartTime(model.taskStartTime);
            taskEndTime(model.taskEndTime);
            s3Bucket(model.s3Bucket);
            s3Prefix(model.s3Prefix);
            iamRoleArn(model.iamRoleArn);
            kmsKeyId(model.kmsKeyId);
            status(model.status);
            percentProgress(model.percentProgress);
            totalExtractedDataInGB(model.totalExtractedDataInGB);
            failureCause(model.failureCause);
            warningMessage(model.warningMessage);
            sourceType(model.sourceType);
        }

        public final String getExportTaskIdentifier() {
            return exportTaskIdentifier;
        }

        public final void setExportTaskIdentifier(String exportTaskIdentifier) {
            this.exportTaskIdentifier = exportTaskIdentifier;
        }

        @Override
        public final Builder exportTaskIdentifier(String exportTaskIdentifier) {
            this.exportTaskIdentifier = exportTaskIdentifier;
            return this;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final Collection<String> getExportOnly() {
            if (exportOnly instanceof SdkAutoConstructList) {
                return null;
            }
            return exportOnly;
        }

        public final void setExportOnly(Collection<String> exportOnly) {
            this.exportOnly = StringListCopier.copy(exportOnly);
        }

        @Override
        public final Builder exportOnly(Collection<String> exportOnly) {
            this.exportOnly = StringListCopier.copy(exportOnly);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exportOnly(String... exportOnly) {
            exportOnly(Arrays.asList(exportOnly));
            return this;
        }

        public final Instant getSnapshotTime() {
            return snapshotTime;
        }

        public final void setSnapshotTime(Instant snapshotTime) {
            this.snapshotTime = snapshotTime;
        }

        @Override
        public final Builder snapshotTime(Instant snapshotTime) {
            this.snapshotTime = snapshotTime;
            return this;
        }

        public final Instant getTaskStartTime() {
            return taskStartTime;
        }

        public final void setTaskStartTime(Instant taskStartTime) {
            this.taskStartTime = taskStartTime;
        }

        @Override
        public final Builder taskStartTime(Instant taskStartTime) {
            this.taskStartTime = taskStartTime;
            return this;
        }

        public final Instant getTaskEndTime() {
            return taskEndTime;
        }

        public final void setTaskEndTime(Instant taskEndTime) {
            this.taskEndTime = taskEndTime;
        }

        @Override
        public final Builder taskEndTime(Instant taskEndTime) {
            this.taskEndTime = taskEndTime;
            return this;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final String getS3Prefix() {
            return s3Prefix;
        }

        public final void setS3Prefix(String s3Prefix) {
            this.s3Prefix = s3Prefix;
        }

        @Override
        public final Builder s3Prefix(String s3Prefix) {
            this.s3Prefix = s3Prefix;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Integer getPercentProgress() {
            return percentProgress;
        }

        public final void setPercentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
        }

        @Override
        public final Builder percentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        public final Integer getTotalExtractedDataInGB() {
            return totalExtractedDataInGB;
        }

        public final void setTotalExtractedDataInGB(Integer totalExtractedDataInGB) {
            this.totalExtractedDataInGB = totalExtractedDataInGB;
        }

        @Override
        public final Builder totalExtractedDataInGB(Integer totalExtractedDataInGB) {
            this.totalExtractedDataInGB = totalExtractedDataInGB;
            return this;
        }

        public final String getFailureCause() {
            return failureCause;
        }

        public final void setFailureCause(String failureCause) {
            this.failureCause = failureCause;
        }

        @Override
        public final Builder failureCause(String failureCause) {
            this.failureCause = failureCause;
            return this;
        }

        public final String getWarningMessage() {
            return warningMessage;
        }

        public final void setWarningMessage(String warningMessage) {
            this.warningMessage = warningMessage;
        }

        @Override
        public final Builder warningMessage(String warningMessage) {
            this.warningMessage = warningMessage;
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public final Builder sourceType(ExportSourceType sourceType) {
            this.sourceType(sourceType == null ? null : sourceType.toString());
            return this;
        }

        @Override
        public ExportTask build() {
            return new ExportTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
