/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The version of the database engine that a DB instance can be upgraded to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpgradeTarget implements SdkPojo, Serializable, ToCopyableBuilder<UpgradeTarget.Builder, UpgradeTarget> {
    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(UpgradeTarget::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(UpgradeTarget::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(UpgradeTarget::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Boolean> AUTO_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoUpgrade").getter(getter(UpgradeTarget::autoUpgrade)).setter(setter(Builder::autoUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoUpgrade").build()).build();

    private static final SdkField<Boolean> IS_MAJOR_VERSION_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsMajorVersionUpgrade").getter(getter(UpgradeTarget::isMajorVersionUpgrade))
            .setter(setter(Builder::isMajorVersionUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsMajorVersionUpgrade").build())
            .build();

    private static final SdkField<List<String>> SUPPORTED_ENGINE_MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedEngineModes")
            .getter(getter(UpgradeTarget::supportedEngineModes))
            .setter(setter(Builder::supportedEngineModes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedEngineModes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPORTS_PARALLEL_QUERY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsParallelQuery").getter(getter(UpgradeTarget::supportsParallelQuery))
            .setter(setter(Builder::supportsParallelQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsParallelQuery").build())
            .build();

    private static final SdkField<Boolean> SUPPORTS_GLOBAL_DATABASES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsGlobalDatabases").getter(getter(UpgradeTarget::supportsGlobalDatabases))
            .setter(setter(Builder::supportsGlobalDatabases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsGlobalDatabases").build())
            .build();

    private static final SdkField<Boolean> SUPPORTS_BABELFISH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsBabelfish").getter(getter(UpgradeTarget::supportsBabelfish))
            .setter(setter(Builder::supportsBabelfish))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsBabelfish").build()).build();

    private static final SdkField<Boolean> SUPPORTS_LOCAL_WRITE_FORWARDING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsLocalWriteForwarding")
            .getter(getter(UpgradeTarget::supportsLocalWriteForwarding))
            .setter(setter(Builder::supportsLocalWriteForwarding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsLocalWriteForwarding")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_FIELD,
            ENGINE_VERSION_FIELD, DESCRIPTION_FIELD, AUTO_UPGRADE_FIELD, IS_MAJOR_VERSION_UPGRADE_FIELD,
            SUPPORTED_ENGINE_MODES_FIELD, SUPPORTS_PARALLEL_QUERY_FIELD, SUPPORTS_GLOBAL_DATABASES_FIELD,
            SUPPORTS_BABELFISH_FIELD, SUPPORTS_LOCAL_WRITE_FORWARDING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String engine;

    private final String engineVersion;

    private final String description;

    private final Boolean autoUpgrade;

    private final Boolean isMajorVersionUpgrade;

    private final List<String> supportedEngineModes;

    private final Boolean supportsParallelQuery;

    private final Boolean supportsGlobalDatabases;

    private final Boolean supportsBabelfish;

    private final Boolean supportsLocalWriteForwarding;

    private UpgradeTarget(BuilderImpl builder) {
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.description = builder.description;
        this.autoUpgrade = builder.autoUpgrade;
        this.isMajorVersionUpgrade = builder.isMajorVersionUpgrade;
        this.supportedEngineModes = builder.supportedEngineModes;
        this.supportsParallelQuery = builder.supportsParallelQuery;
        this.supportsGlobalDatabases = builder.supportsGlobalDatabases;
        this.supportsBabelfish = builder.supportsBabelfish;
        this.supportsLocalWriteForwarding = builder.supportsLocalWriteForwarding;
    }

    /**
     * <p>
     * The name of the upgrade target database engine.
     * </p>
     * 
     * @return The name of the upgrade target database engine.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The version number of the upgrade target database engine.
     * </p>
     * 
     * @return The version number of the upgrade target database engine.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The version of the database engine that a DB instance can be upgraded to.
     * </p>
     * 
     * @return The version of the database engine that a DB instance can be upgraded to.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Indicates whether the target version is applied to any source DB instances that have
     * <code>AutoMinorVersionUpgrade</code> set to true.
     * </p>
     * 
     * @return Indicates whether the target version is applied to any source DB instances that have
     *         <code>AutoMinorVersionUpgrade</code> set to true.
     */
    public final Boolean autoUpgrade() {
        return autoUpgrade;
    }

    /**
     * <p>
     * Indicates whether upgrading to the target version requires upgrading the major version of the database engine.
     * </p>
     * 
     * @return Indicates whether upgrading to the target version requires upgrading the major version of the database
     *         engine.
     */
    public final Boolean isMajorVersionUpgrade() {
        return isMajorVersionUpgrade;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedEngineModes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedEngineModes() {
        return supportedEngineModes != null && !(supportedEngineModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the supported DB engine modes for the target engine version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedEngineModes} method.
     * </p>
     * 
     * @return A list of the supported DB engine modes for the target engine version.
     */
    public final List<String> supportedEngineModes() {
        return supportedEngineModes;
    }

    /**
     * <p>
     * Indicates whether you can use Aurora parallel query with the target engine version.
     * </p>
     * 
     * @return Indicates whether you can use Aurora parallel query with the target engine version.
     */
    public final Boolean supportsParallelQuery() {
        return supportsParallelQuery;
    }

    /**
     * <p>
     * Indicates whether you can use Aurora global databases with the target engine version.
     * </p>
     * 
     * @return Indicates whether you can use Aurora global databases with the target engine version.
     */
    public final Boolean supportsGlobalDatabases() {
        return supportsGlobalDatabases;
    }

    /**
     * <p>
     * Indicates whether you can use Babelfish for Aurora PostgreSQL with the target engine version.
     * </p>
     * 
     * @return Indicates whether you can use Babelfish for Aurora PostgreSQL with the target engine version.
     */
    public final Boolean supportsBabelfish() {
        return supportsBabelfish;
    }

    /**
     * <p>
     * Indicates whether the target engine version supports forwarding write operations from reader DB instances to the
     * writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB instances.
     * </p>
     * <p>
     * Valid for: Aurora DB clusters only
     * </p>
     * 
     * @return Indicates whether the target engine version supports forwarding write operations from reader DB instances
     *         to the writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB
     *         instances.</p>
     *         <p>
     *         Valid for: Aurora DB clusters only
     */
    public final Boolean supportsLocalWriteForwarding() {
        return supportsLocalWriteForwarding;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(autoUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(isMajorVersionUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedEngineModes() ? supportedEngineModes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(supportsParallelQuery());
        hashCode = 31 * hashCode + Objects.hashCode(supportsGlobalDatabases());
        hashCode = 31 * hashCode + Objects.hashCode(supportsBabelfish());
        hashCode = 31 * hashCode + Objects.hashCode(supportsLocalWriteForwarding());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpgradeTarget)) {
            return false;
        }
        UpgradeTarget other = (UpgradeTarget) obj;
        return Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(description(), other.description()) && Objects.equals(autoUpgrade(), other.autoUpgrade())
                && Objects.equals(isMajorVersionUpgrade(), other.isMajorVersionUpgrade())
                && hasSupportedEngineModes() == other.hasSupportedEngineModes()
                && Objects.equals(supportedEngineModes(), other.supportedEngineModes())
                && Objects.equals(supportsParallelQuery(), other.supportsParallelQuery())
                && Objects.equals(supportsGlobalDatabases(), other.supportsGlobalDatabases())
                && Objects.equals(supportsBabelfish(), other.supportsBabelfish())
                && Objects.equals(supportsLocalWriteForwarding(), other.supportsLocalWriteForwarding());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpgradeTarget").add("Engine", engine()).add("EngineVersion", engineVersion())
                .add("Description", description()).add("AutoUpgrade", autoUpgrade())
                .add("IsMajorVersionUpgrade", isMajorVersionUpgrade())
                .add("SupportedEngineModes", hasSupportedEngineModes() ? supportedEngineModes() : null)
                .add("SupportsParallelQuery", supportsParallelQuery()).add("SupportsGlobalDatabases", supportsGlobalDatabases())
                .add("SupportsBabelfish", supportsBabelfish())
                .add("SupportsLocalWriteForwarding", supportsLocalWriteForwarding()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AutoUpgrade":
            return Optional.ofNullable(clazz.cast(autoUpgrade()));
        case "IsMajorVersionUpgrade":
            return Optional.ofNullable(clazz.cast(isMajorVersionUpgrade()));
        case "SupportedEngineModes":
            return Optional.ofNullable(clazz.cast(supportedEngineModes()));
        case "SupportsParallelQuery":
            return Optional.ofNullable(clazz.cast(supportsParallelQuery()));
        case "SupportsGlobalDatabases":
            return Optional.ofNullable(clazz.cast(supportsGlobalDatabases()));
        case "SupportsBabelfish":
            return Optional.ofNullable(clazz.cast(supportsBabelfish()));
        case "SupportsLocalWriteForwarding":
            return Optional.ofNullable(clazz.cast(supportsLocalWriteForwarding()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpgradeTarget, T> g) {
        return obj -> g.apply((UpgradeTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UpgradeTarget> {
        /**
         * <p>
         * The name of the upgrade target database engine.
         * </p>
         * 
         * @param engine
         *        The name of the upgrade target database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The version number of the upgrade target database engine.
         * </p>
         * 
         * @param engineVersion
         *        The version number of the upgrade target database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The version of the database engine that a DB instance can be upgraded to.
         * </p>
         * 
         * @param description
         *        The version of the database engine that a DB instance can be upgraded to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Indicates whether the target version is applied to any source DB instances that have
         * <code>AutoMinorVersionUpgrade</code> set to true.
         * </p>
         * 
         * @param autoUpgrade
         *        Indicates whether the target version is applied to any source DB instances that have
         *        <code>AutoMinorVersionUpgrade</code> set to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoUpgrade(Boolean autoUpgrade);

        /**
         * <p>
         * Indicates whether upgrading to the target version requires upgrading the major version of the database
         * engine.
         * </p>
         * 
         * @param isMajorVersionUpgrade
         *        Indicates whether upgrading to the target version requires upgrading the major version of the database
         *        engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isMajorVersionUpgrade(Boolean isMajorVersionUpgrade);

        /**
         * <p>
         * A list of the supported DB engine modes for the target engine version.
         * </p>
         * 
         * @param supportedEngineModes
         *        A list of the supported DB engine modes for the target engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedEngineModes(Collection<String> supportedEngineModes);

        /**
         * <p>
         * A list of the supported DB engine modes for the target engine version.
         * </p>
         * 
         * @param supportedEngineModes
         *        A list of the supported DB engine modes for the target engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedEngineModes(String... supportedEngineModes);

        /**
         * <p>
         * Indicates whether you can use Aurora parallel query with the target engine version.
         * </p>
         * 
         * @param supportsParallelQuery
         *        Indicates whether you can use Aurora parallel query with the target engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsParallelQuery(Boolean supportsParallelQuery);

        /**
         * <p>
         * Indicates whether you can use Aurora global databases with the target engine version.
         * </p>
         * 
         * @param supportsGlobalDatabases
         *        Indicates whether you can use Aurora global databases with the target engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsGlobalDatabases(Boolean supportsGlobalDatabases);

        /**
         * <p>
         * Indicates whether you can use Babelfish for Aurora PostgreSQL with the target engine version.
         * </p>
         * 
         * @param supportsBabelfish
         *        Indicates whether you can use Babelfish for Aurora PostgreSQL with the target engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsBabelfish(Boolean supportsBabelfish);

        /**
         * <p>
         * Indicates whether the target engine version supports forwarding write operations from reader DB instances to
         * the writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB instances.
         * </p>
         * <p>
         * Valid for: Aurora DB clusters only
         * </p>
         * 
         * @param supportsLocalWriteForwarding
         *        Indicates whether the target engine version supports forwarding write operations from reader DB
         *        instances to the writer DB instance in the DB cluster. By default, write operations aren't allowed on
         *        reader DB instances.</p>
         *        <p>
         *        Valid for: Aurora DB clusters only
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsLocalWriteForwarding(Boolean supportsLocalWriteForwarding);
    }

    static final class BuilderImpl implements Builder {
        private String engine;

        private String engineVersion;

        private String description;

        private Boolean autoUpgrade;

        private Boolean isMajorVersionUpgrade;

        private List<String> supportedEngineModes = DefaultSdkAutoConstructList.getInstance();

        private Boolean supportsParallelQuery;

        private Boolean supportsGlobalDatabases;

        private Boolean supportsBabelfish;

        private Boolean supportsLocalWriteForwarding;

        private BuilderImpl() {
        }

        private BuilderImpl(UpgradeTarget model) {
            engine(model.engine);
            engineVersion(model.engineVersion);
            description(model.description);
            autoUpgrade(model.autoUpgrade);
            isMajorVersionUpgrade(model.isMajorVersionUpgrade);
            supportedEngineModes(model.supportedEngineModes);
            supportsParallelQuery(model.supportsParallelQuery);
            supportsGlobalDatabases(model.supportsGlobalDatabases);
            supportsBabelfish(model.supportsBabelfish);
            supportsLocalWriteForwarding(model.supportsLocalWriteForwarding);
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Boolean getAutoUpgrade() {
            return autoUpgrade;
        }

        public final void setAutoUpgrade(Boolean autoUpgrade) {
            this.autoUpgrade = autoUpgrade;
        }

        @Override
        public final Builder autoUpgrade(Boolean autoUpgrade) {
            this.autoUpgrade = autoUpgrade;
            return this;
        }

        public final Boolean getIsMajorVersionUpgrade() {
            return isMajorVersionUpgrade;
        }

        public final void setIsMajorVersionUpgrade(Boolean isMajorVersionUpgrade) {
            this.isMajorVersionUpgrade = isMajorVersionUpgrade;
        }

        @Override
        public final Builder isMajorVersionUpgrade(Boolean isMajorVersionUpgrade) {
            this.isMajorVersionUpgrade = isMajorVersionUpgrade;
            return this;
        }

        public final Collection<String> getSupportedEngineModes() {
            if (supportedEngineModes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedEngineModes;
        }

        public final void setSupportedEngineModes(Collection<String> supportedEngineModes) {
            this.supportedEngineModes = EngineModeListCopier.copy(supportedEngineModes);
        }

        @Override
        public final Builder supportedEngineModes(Collection<String> supportedEngineModes) {
            this.supportedEngineModes = EngineModeListCopier.copy(supportedEngineModes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedEngineModes(String... supportedEngineModes) {
            supportedEngineModes(Arrays.asList(supportedEngineModes));
            return this;
        }

        public final Boolean getSupportsParallelQuery() {
            return supportsParallelQuery;
        }

        public final void setSupportsParallelQuery(Boolean supportsParallelQuery) {
            this.supportsParallelQuery = supportsParallelQuery;
        }

        @Override
        public final Builder supportsParallelQuery(Boolean supportsParallelQuery) {
            this.supportsParallelQuery = supportsParallelQuery;
            return this;
        }

        public final Boolean getSupportsGlobalDatabases() {
            return supportsGlobalDatabases;
        }

        public final void setSupportsGlobalDatabases(Boolean supportsGlobalDatabases) {
            this.supportsGlobalDatabases = supportsGlobalDatabases;
        }

        @Override
        public final Builder supportsGlobalDatabases(Boolean supportsGlobalDatabases) {
            this.supportsGlobalDatabases = supportsGlobalDatabases;
            return this;
        }

        public final Boolean getSupportsBabelfish() {
            return supportsBabelfish;
        }

        public final void setSupportsBabelfish(Boolean supportsBabelfish) {
            this.supportsBabelfish = supportsBabelfish;
        }

        @Override
        public final Builder supportsBabelfish(Boolean supportsBabelfish) {
            this.supportsBabelfish = supportsBabelfish;
            return this;
        }

        public final Boolean getSupportsLocalWriteForwarding() {
            return supportsLocalWriteForwarding;
        }

        public final void setSupportsLocalWriteForwarding(Boolean supportsLocalWriteForwarding) {
            this.supportsLocalWriteForwarding = supportsLocalWriteForwarding;
        }

        @Override
        public final Builder supportsLocalWriteForwarding(Boolean supportsLocalWriteForwarding) {
            this.supportsLocalWriteForwarding = supportsLocalWriteForwarding;
            return this;
        }

        @Override
        public UpgradeTarget build() {
            return new UpgradeTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
