/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshiftdata;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.redshiftdata.model.ActiveStatementsExceededException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DatabaseConnectionException;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableResponse;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultRequest;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultResponse;
import software.amazon.awssdk.services.redshiftdata.model.InternalServerException;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesResponse;
import software.amazon.awssdk.services.redshiftdata.model.RedshiftDataException;
import software.amazon.awssdk.services.redshiftdata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.redshiftdata.model.ValidationException;
import software.amazon.awssdk.services.redshiftdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.CancelStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.GetStatementResultRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListSchemasRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListStatementsRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RedshiftDataClient}.
 *
 * @see RedshiftDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRedshiftDataClient implements RedshiftDataClient {
    private static final Logger log = Logger.loggerFor(DefaultRedshiftDataClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final RedshiftDataServiceClientConfiguration serviceClientConfiguration;

    protected DefaultRedshiftDataClient(RedshiftDataServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Runs one or more SQL statements, which can be data manipulation language (DML) or data definition language (DDL).
     * Depending on the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param batchExecuteStatementRequest
     * @return Result of the BatchExecuteStatement operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws ActiveStatementsExceededException
     *         The number of active statements exceeds the limit.
     * @throws BatchExecuteStatementException
     *         An SQL statement encountered an environmental error while running.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchExecuteStatementResponse batchExecuteStatement(BatchExecuteStatementRequest batchExecuteStatementRequest)
            throws ValidationException, ActiveStatementsExceededException, BatchExecuteStatementException, AwsServiceException,
            SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                    .withOperationName("BatchExecuteStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchExecuteStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a running query. To be canceled, a query must be running.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param cancelStatementRequest
     * @return Result of the CancelStatement operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws ResourceNotFoundException
     *         The Amazon Redshift Data API operation failed due to a missing resource.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws DatabaseConnectionException
     *         Connection to a database failed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.CancelStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/CancelStatement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelStatementResponse cancelStatement(CancelStatementRequest cancelStatementRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, DatabaseConnectionException, AwsServiceException,
            SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelStatement");

            return clientHandler.execute(new ClientExecutionParams<CancelStatementRequest, CancelStatementResponse>()
                    .withOperationName("CancelStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the details about a specific instance when a query was run by the Amazon Redshift Data API. The
     * information includes when the query started, when it finished, the query status, the number of rows returned, and
     * the SQL statement.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param describeStatementRequest
     * @return Result of the DescribeStatement operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws ResourceNotFoundException
     *         The Amazon Redshift Data API operation failed due to a missing resource.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.DescribeStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStatementResponse describeStatement(DescribeStatementRequest describeStatementRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStatement");

            return clientHandler.execute(new ClientExecutionParams<DescribeStatementRequest, DescribeStatementResponse>()
                    .withOperationName("DescribeStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the detailed information about a table from metadata in the cluster. The information includes its
     * columns. A token is returned to page through the column list. Depending on the authorization method, use one of
     * the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param describeTableRequest
     * @return Result of the DescribeTable operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws DatabaseConnectionException
     *         Connection to a database failed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeTableResponse describeTable(DescribeTableRequest describeTableRequest) throws ValidationException,
            InternalServerException, DatabaseConnectionException, AwsServiceException, SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");

            return clientHandler.execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                    .withOperationName("DescribeTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs an SQL statement, which can be data manipulation language (DML) or data definition language (DDL). This
     * statement must be a single SQL statement. Depending on the authorization method, use one of the following
     * combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param executeStatementRequest
     * @return Result of the ExecuteStatement operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws ExecuteStatementException
     *         The SQL statement encountered an environmental error while running.
     * @throws ActiveStatementsExceededException
     *         The number of active statements exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExecuteStatementResponse executeStatement(ExecuteStatementRequest executeStatementRequest) throws ValidationException,
            ExecuteStatementException, ActiveStatementsExceededException, AwsServiceException, SdkClientException,
            RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                    .withOperationName("ExecuteStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(executeStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Fetches the temporarily cached result of an SQL statement. A token is returned to page through the statement
     * results.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param getStatementResultRequest
     * @return Result of the GetStatementResult operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws ResourceNotFoundException
     *         The Amazon Redshift Data API operation failed due to a missing resource.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.GetStatementResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/GetStatementResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetStatementResultResponse getStatementResult(GetStatementResultRequest getStatementResultRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStatementResultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetStatementResultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStatementResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStatementResult");

            return clientHandler.execute(new ClientExecutionParams<GetStatementResultRequest, GetStatementResultResponse>()
                    .withOperationName("GetStatementResult").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getStatementResultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStatementResultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the databases in a cluster. A token is returned to page through the database list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listDatabasesRequest
     * @return Result of the ListDatabases operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws DatabaseConnectionException
     *         Connection to a database failed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListDatabases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDatabasesResponse listDatabases(ListDatabasesRequest listDatabasesRequest) throws ValidationException,
            InternalServerException, DatabaseConnectionException, AwsServiceException, SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");

            return clientHandler.execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                    .withOperationName("ListDatabases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatabasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the schemas in a database. A token is returned to page through the schema list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listSchemasRequest
     * @return Result of the ListSchemas operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws DatabaseConnectionException
     *         Connection to a database failed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.ListSchemas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSchemasResponse listSchemas(ListSchemasRequest listSchemasRequest) throws ValidationException,
            InternalServerException, DatabaseConnectionException, AwsServiceException, SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSchemasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemas");

            return clientHandler.execute(new ClientExecutionParams<ListSchemasRequest, ListSchemasResponse>()
                    .withOperationName("ListSchemas").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSchemasRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSchemasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List of SQL statements. By default, only finished statements are shown. A token is returned to page through the
     * statement list.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listStatementsRequest
     * @return Result of the ListStatements operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.ListStatements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListStatements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStatementsResponse listStatements(ListStatementsRequest listStatementsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStatementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStatementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStatementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStatements");

            return clientHandler.execute(new ClientExecutionParams<ListStatementsRequest, ListStatementsResponse>()
                    .withOperationName("ListStatements").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStatementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStatementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tables in a database. If neither <code>SchemaPattern</code> nor <code>TablePattern</code> are specified,
     * then all tables in the database are returned. A token is returned to page through the table list. Depending on
     * the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listTablesRequest
     * @return Result of the ListTables operation returned by the service.
     * @throws ValidationException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws InternalServerException
     *         The Amazon Redshift Data API operation failed due to invalid input.
     * @throws DatabaseConnectionException
     *         Connection to a database failed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RedshiftDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RedshiftDataClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws ValidationException,
            InternalServerException, DatabaseConnectionException, AwsServiceException, SdkClientException, RedshiftDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTablesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RedshiftDataException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveStatementsExceededException")
                                .exceptionBuilderSupplier(ActiveStatementsExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseConnectionException")
                                .exceptionBuilderSupplier(DatabaseConnectionException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecuteStatementException")
                                .exceptionBuilderSupplier(ExecuteStatementException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchExecuteStatementException")
                                .exceptionBuilderSupplier(BatchExecuteStatementException::builder).httpStatusCode(500).build());
    }

    @Override
    public final RedshiftDataServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
