/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A channel is a named input source that training algorithms can consume. This channel is used for AutoML jobs V2 (jobs
 * created by calling <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAutoMLJobV2.html">CreateAutoMLJobV2</a>).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoMLJobChannel implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoMLJobChannel.Builder, AutoMLJobChannel> {
    private static final SdkField<String> CHANNEL_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChannelType").getter(getter(AutoMLJobChannel::channelTypeAsString)).setter(setter(Builder::channelType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChannelType").build()).build();

    private static final SdkField<String> CONTENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentType").getter(getter(AutoMLJobChannel::contentType)).setter(setter(Builder::contentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentType").build()).build();

    private static final SdkField<String> COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompressionType").getter(getter(AutoMLJobChannel::compressionTypeAsString))
            .setter(setter(Builder::compressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompressionType").build()).build();

    private static final SdkField<AutoMLDataSource> DATA_SOURCE_FIELD = SdkField
            .<AutoMLDataSource> builder(MarshallingType.SDK_POJO).memberName("DataSource")
            .getter(getter(AutoMLJobChannel::dataSource)).setter(setter(Builder::dataSource))
            .constructor(AutoMLDataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANNEL_TYPE_FIELD,
            CONTENT_TYPE_FIELD, COMPRESSION_TYPE_FIELD, DATA_SOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String channelType;

    private final String contentType;

    private final String compressionType;

    private final AutoMLDataSource dataSource;

    private AutoMLJobChannel(BuilderImpl builder) {
        this.channelType = builder.channelType;
        this.contentType = builder.contentType;
        this.compressionType = builder.compressionType;
        this.dataSource = builder.dataSource;
    }

    /**
     * <p>
     * The type of channel. Defines whether the data are used for training or validation. The default value is
     * <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
     * <code>ContentType</code>
     * </p>
     * <note>
     * <p>
     * The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channelType} will
     * return {@link AutoMLChannelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelTypeAsString}.
     * </p>
     * 
     * @return The type of channel. Defines whether the data are used for training or validation. The default value is
     *         <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
     *         <code>ContentType</code> </p> <note>
     *         <p>
     *         The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
     *         </p>
     * @see AutoMLChannelType
     */
    public final AutoMLChannelType channelType() {
        return AutoMLChannelType.fromValue(channelType);
    }

    /**
     * <p>
     * The type of channel. Defines whether the data are used for training or validation. The default value is
     * <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
     * <code>ContentType</code>
     * </p>
     * <note>
     * <p>
     * The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channelType} will
     * return {@link AutoMLChannelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelTypeAsString}.
     * </p>
     * 
     * @return The type of channel. Defines whether the data are used for training or validation. The default value is
     *         <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
     *         <code>ContentType</code> </p> <note>
     *         <p>
     *         The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
     *         </p>
     * @see AutoMLChannelType
     */
    public final String channelTypeAsString() {
        return channelType;
    }

    /**
     * <p>
     * The content type of the data from the input source. The following are the allowed content types for different
     * problems:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For tabular problem types: <code>text/csv;header=present</code> or <code>x-application/vnd.amazon+parquet</code>.
     * The default value is <code>text/csv;header=present</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For image classification: <code>image/png</code>, <code>image/jpeg</code>, or <code>image/*</code>. The default
     * value is <code>image/*</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For text classification: <code>text/csv;header=present</code> or <code>x-application/vnd.amazon+parquet</code>.
     * The default value is <code>text/csv;header=present</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For time-series forecasting: <code>text/csv;header=present</code> or
     * <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The content type of the data from the input source. The following are the allowed content types for
     *         different problems:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For tabular problem types: <code>text/csv;header=present</code> or
     *         <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For image classification: <code>image/png</code>, <code>image/jpeg</code>, or <code>image/*</code>. The
     *         default value is <code>image/*</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For text classification: <code>text/csv;header=present</code> or
     *         <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For time-series forecasting: <code>text/csv;header=present</code> or
     *         <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
     *         </p>
     *         </li>
     */
    public final String contentType() {
        return contentType;
    }

    /**
     * <p>
     * The allowed compression types depend on the input format and problem type. We allow the compression type
     * <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the compression
     * type should be <code>None</code>. If no compression type is provided, we default to <code>None</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link CompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return The allowed compression types depend on the input format and problem type. We allow the compression type
     *         <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the
     *         compression type should be <code>None</code>. If no compression type is provided, we default to
     *         <code>None</code>.
     * @see CompressionType
     */
    public final CompressionType compressionType() {
        return CompressionType.fromValue(compressionType);
    }

    /**
     * <p>
     * The allowed compression types depend on the input format and problem type. We allow the compression type
     * <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the compression
     * type should be <code>None</code>. If no compression type is provided, we default to <code>None</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link CompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return The allowed compression types depend on the input format and problem type. We allow the compression type
     *         <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the
     *         compression type should be <code>None</code>. If no compression type is provided, we default to
     *         <code>None</code>.
     * @see CompressionType
     */
    public final String compressionTypeAsString() {
        return compressionType;
    }

    /**
     * <p>
     * The data source for an AutoML channel (Required).
     * </p>
     * 
     * @return The data source for an AutoML channel (Required).
     */
    public final AutoMLDataSource dataSource() {
        return dataSource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(channelTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(contentType());
        hashCode = 31 * hashCode + Objects.hashCode(compressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dataSource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoMLJobChannel)) {
            return false;
        }
        AutoMLJobChannel other = (AutoMLJobChannel) obj;
        return Objects.equals(channelTypeAsString(), other.channelTypeAsString())
                && Objects.equals(contentType(), other.contentType())
                && Objects.equals(compressionTypeAsString(), other.compressionTypeAsString())
                && Objects.equals(dataSource(), other.dataSource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoMLJobChannel").add("ChannelType", channelTypeAsString()).add("ContentType", contentType())
                .add("CompressionType", compressionTypeAsString()).add("DataSource", dataSource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChannelType":
            return Optional.ofNullable(clazz.cast(channelTypeAsString()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentType()));
        case "CompressionType":
            return Optional.ofNullable(clazz.cast(compressionTypeAsString()));
        case "DataSource":
            return Optional.ofNullable(clazz.cast(dataSource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoMLJobChannel, T> g) {
        return obj -> g.apply((AutoMLJobChannel) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoMLJobChannel> {
        /**
         * <p>
         * The type of channel. Defines whether the data are used for training or validation. The default value is
         * <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
         * <code>ContentType</code>
         * </p>
         * <note>
         * <p>
         * The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
         * </p>
         * </note>
         * 
         * @param channelType
         *        The type of channel. Defines whether the data are used for training or validation. The default value
         *        is <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share
         *        the same <code>ContentType</code> </p> <note>
         *        <p>
         *        The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
         *        </p>
         * @see AutoMLChannelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLChannelType
         */
        Builder channelType(String channelType);

        /**
         * <p>
         * The type of channel. Defines whether the data are used for training or validation. The default value is
         * <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share the same
         * <code>ContentType</code>
         * </p>
         * <note>
         * <p>
         * The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
         * </p>
         * </note>
         * 
         * @param channelType
         *        The type of channel. Defines whether the data are used for training or validation. The default value
         *        is <code>training</code>. Channels for <code>training</code> and <code>validation</code> must share
         *        the same <code>ContentType</code> </p> <note>
         *        <p>
         *        The type of channel defaults to <code>training</code> for the time-series forecasting problem type.
         *        </p>
         * @see AutoMLChannelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLChannelType
         */
        Builder channelType(AutoMLChannelType channelType);

        /**
         * <p>
         * The content type of the data from the input source. The following are the allowed content types for different
         * problems:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For tabular problem types: <code>text/csv;header=present</code> or
         * <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For image classification: <code>image/png</code>, <code>image/jpeg</code>, or <code>image/*</code>. The
         * default value is <code>image/*</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For text classification: <code>text/csv;header=present</code> or
         * <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For time-series forecasting: <code>text/csv;header=present</code> or
         * <code>x-application/vnd.amazon+parquet</code>. The default value is <code>text/csv;header=present</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param contentType
         *        The content type of the data from the input source. The following are the allowed content types for
         *        different problems:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For tabular problem types: <code>text/csv;header=present</code> or
         *        <code>x-application/vnd.amazon+parquet</code>. The default value is
         *        <code>text/csv;header=present</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For image classification: <code>image/png</code>, <code>image/jpeg</code>, or <code>image/*</code>.
         *        The default value is <code>image/*</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For text classification: <code>text/csv;header=present</code> or
         *        <code>x-application/vnd.amazon+parquet</code>. The default value is
         *        <code>text/csv;header=present</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For time-series forecasting: <code>text/csv;header=present</code> or
         *        <code>x-application/vnd.amazon+parquet</code>. The default value is
         *        <code>text/csv;header=present</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The allowed compression types depend on the input format and problem type. We allow the compression type
         * <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the
         * compression type should be <code>None</code>. If no compression type is provided, we default to
         * <code>None</code>.
         * </p>
         * 
         * @param compressionType
         *        The allowed compression types depend on the input format and problem type. We allow the compression
         *        type <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs,
         *        the compression type should be <code>None</code>. If no compression type is provided, we default to
         *        <code>None</code>.
         * @see CompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionType
         */
        Builder compressionType(String compressionType);

        /**
         * <p>
         * The allowed compression types depend on the input format and problem type. We allow the compression type
         * <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs, the
         * compression type should be <code>None</code>. If no compression type is provided, we default to
         * <code>None</code>.
         * </p>
         * 
         * @param compressionType
         *        The allowed compression types depend on the input format and problem type. We allow the compression
         *        type <code>Gzip</code> for <code>S3Prefix</code> inputs on tabular data only. For all other inputs,
         *        the compression type should be <code>None</code>. If no compression type is provided, we default to
         *        <code>None</code>.
         * @see CompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionType
         */
        Builder compressionType(CompressionType compressionType);

        /**
         * <p>
         * The data source for an AutoML channel (Required).
         * </p>
         * 
         * @param dataSource
         *        The data source for an AutoML channel (Required).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSource(AutoMLDataSource dataSource);

        /**
         * <p>
         * The data source for an AutoML channel (Required).
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoMLDataSource.Builder} avoiding the
         * need to create one manually via {@link AutoMLDataSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLDataSource.Builder#build()} is called immediately and its
         * result is passed to {@link #dataSource(AutoMLDataSource)}.
         * 
         * @param dataSource
         *        a consumer that will call methods on {@link AutoMLDataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSource(AutoMLDataSource)
         */
        default Builder dataSource(Consumer<AutoMLDataSource.Builder> dataSource) {
            return dataSource(AutoMLDataSource.builder().applyMutation(dataSource).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String channelType;

        private String contentType;

        private String compressionType;

        private AutoMLDataSource dataSource;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoMLJobChannel model) {
            channelType(model.channelType);
            contentType(model.contentType);
            compressionType(model.compressionType);
            dataSource(model.dataSource);
        }

        public final String getChannelType() {
            return channelType;
        }

        public final void setChannelType(String channelType) {
            this.channelType = channelType;
        }

        @Override
        public final Builder channelType(String channelType) {
            this.channelType = channelType;
            return this;
        }

        @Override
        public final Builder channelType(AutoMLChannelType channelType) {
            this.channelType(channelType == null ? null : channelType.toString());
            return this;
        }

        public final String getContentType() {
            return contentType;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        public final String getCompressionType() {
            return compressionType;
        }

        public final void setCompressionType(String compressionType) {
            this.compressionType = compressionType;
        }

        @Override
        public final Builder compressionType(String compressionType) {
            this.compressionType = compressionType;
            return this;
        }

        @Override
        public final Builder compressionType(CompressionType compressionType) {
            this.compressionType(compressionType == null ? null : compressionType.toString());
            return this;
        }

        public final AutoMLDataSource.Builder getDataSource() {
            return dataSource != null ? dataSource.toBuilder() : null;
        }

        public final void setDataSource(AutoMLDataSource.BuilderImpl dataSource) {
            this.dataSource = dataSource != null ? dataSource.build() : null;
        }

        @Override
        public final Builder dataSource(AutoMLDataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        @Override
        public AutoMLJobChannel build() {
            return new AutoMLJobChannel(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
