/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the Route 53 DNS records that you want Cloud Map to create when you
 * register an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DnsRecord implements SdkPojo, Serializable, ToCopyableBuilder<DnsRecord.Builder, DnsRecord> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DnsRecord::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Long> TTL_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("TTL")
            .getter(getter(DnsRecord::ttl)).setter(setter(Builder::ttl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TTL").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, TTL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final Long ttl;

    private DnsRecord(BuilderImpl builder) {
        this.type = builder.type;
        this.ttl = builder.ttl;
    }

    /**
     * <p>
     * The type of the resource, which indicates the type of value that Route 53 returns in response to DNS queries. You
     * can specify values for <code>Type</code> in the following combinations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>A</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>AAAA</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>SRV</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CNAME</code> </b>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you want Cloud Map to create a Route 53 alias record when you register an instance, specify <code>A</code> or
     * <code>AAAA</code> for <code>Type</code>.
     * </p>
     * <p>
     * You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when you
     * register an instance. For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     * </p>
     * <p>
     * The following values are supported:
     * </p>
     * <dl>
     * <dt>A</dt>
     * <dd>
     * <p>
     * Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
     * </p>
     * </dd>
     * <dt>AAAA</dt>
     * <dd>
     * <p>
     * Route 53 returns the IP address of the resource in IPv6 format, such as 2001:0db8:85a3:0000:0000:abcd:0001:2345.
     * </p>
     * </dd>
     * <dt>CNAME</dt>
     * <dd>
     * <p>
     * Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You specify the domain name that you want to route traffic to when you register an instance. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
     * >Attributes</a> in the topic <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't specify both <code>CNAME</code> for <code>Type</code> and settings for <code>HealthCheckConfig</code>.
     * If you do, the request will fail with an <code>InvalidInput</code> error.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>SRV</dt>
     * <dd>
     * <p>
     * Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses the
     * following values:
     * </p>
     * <p>
     * <code>priority weight port service-hostname</code>
     * </p>
     * <p>
     * Note the following about the values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be changed.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>port</code> comes from the value that you specify for the <code>AWS_INSTANCE_PORT</code>
     * attribute when you submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>service-hostname</code> is a concatenation of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The value that you specify for <code>InstanceId</code> when you register an instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name of the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name of the namespace.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
     * <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
     * <code>service-hostname</code> is the following:
     * </p>
     * <p>
     * <code>test.backend.example.com</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify settings for an <code>SRV</code> record, note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
     * <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or <code>AAAA</code>
     * records that have the same name as the value of <code>service-hostname</code> in the <code>SRV</code> record. You
     * can ignore these records.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a href=
     * "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
     * >Name</a> element in the documentation about <code>CreateService</code> for information about how to specify the
     * correct name format.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RecordType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the resource, which indicates the type of value that Route 53 returns in response to DNS
     *         queries. You can specify values for <code>Type</code> in the following combinations:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>A</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>AAAA</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>SRV</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CNAME</code> </b>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you want Cloud Map to create a Route 53 alias record when you register an instance, specify
     *         <code>A</code> or <code>AAAA</code> for <code>Type</code>.
     *         </p>
     *         <p>
     *         You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when
     *         you register an instance. For more information, see <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     *         </p>
     *         <p>
     *         The following values are supported:
     *         </p>
     *         <dl>
     *         <dt>A</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
     *         </p>
     *         </dd>
     *         <dt>AAAA</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the IP address of the resource in IPv6 format, such as
     *         2001:0db8:85a3:0000:0000:abcd:0001:2345.
     *         </p>
     *         </dd>
     *         <dt>CNAME</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specify the domain name that you want to route traffic to when you register an instance. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
     *         >Attributes</a> in the topic <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
     *         <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code> error.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>SRV</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses
     *         the following values:
     *         </p>
     *         <p>
     *         <code>priority weight port service-hostname</code>
     *         </p>
     *         <p>
     *         Note the following about the values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be
     *         changed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>port</code> comes from the value that you specify for the
     *         <code>AWS_INSTANCE_PORT</code> attribute when you submit a <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
     *         request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>service-hostname</code> is a concatenation of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The value that you specify for <code>InstanceId</code> when you register an instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name of the service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name of the namespace.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
     *         <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
     *         <code>service-hostname</code> is the following:
     *         </p>
     *         <p>
     *         <code>test.backend.example.com</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify settings for an <code>SRV</code> record, note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
     *         <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
     *         <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
     *         <code>SRV</code> record. You can ignore these records.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a
     *         href=
     *         "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
     *         >Name</a> element in the documentation about <code>CreateService</code> for information about how to
     *         specify the correct name format.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     * @see RecordType
     */
    public final RecordType type() {
        return RecordType.fromValue(type);
    }

    /**
     * <p>
     * The type of the resource, which indicates the type of value that Route 53 returns in response to DNS queries. You
     * can specify values for <code>Type</code> in the following combinations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>A</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>AAAA</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>SRV</code> </b>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CNAME</code> </b>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you want Cloud Map to create a Route 53 alias record when you register an instance, specify <code>A</code> or
     * <code>AAAA</code> for <code>Type</code>.
     * </p>
     * <p>
     * You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when you
     * register an instance. For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     * </p>
     * <p>
     * The following values are supported:
     * </p>
     * <dl>
     * <dt>A</dt>
     * <dd>
     * <p>
     * Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
     * </p>
     * </dd>
     * <dt>AAAA</dt>
     * <dd>
     * <p>
     * Route 53 returns the IP address of the resource in IPv6 format, such as 2001:0db8:85a3:0000:0000:abcd:0001:2345.
     * </p>
     * </dd>
     * <dt>CNAME</dt>
     * <dd>
     * <p>
     * Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You specify the domain name that you want to route traffic to when you register an instance. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
     * >Attributes</a> in the topic <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't specify both <code>CNAME</code> for <code>Type</code> and settings for <code>HealthCheckConfig</code>.
     * If you do, the request will fail with an <code>InvalidInput</code> error.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>SRV</dt>
     * <dd>
     * <p>
     * Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses the
     * following values:
     * </p>
     * <p>
     * <code>priority weight port service-hostname</code>
     * </p>
     * <p>
     * Note the following about the values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be changed.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>port</code> comes from the value that you specify for the <code>AWS_INSTANCE_PORT</code>
     * attribute when you submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>service-hostname</code> is a concatenation of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The value that you specify for <code>InstanceId</code> when you register an instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name of the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name of the namespace.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
     * <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
     * <code>service-hostname</code> is the following:
     * </p>
     * <p>
     * <code>test.backend.example.com</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify settings for an <code>SRV</code> record, note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
     * <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or <code>AAAA</code>
     * records that have the same name as the value of <code>service-hostname</code> in the <code>SRV</code> record. You
     * can ignore these records.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a href=
     * "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
     * >Name</a> element in the documentation about <code>CreateService</code> for information about how to specify the
     * correct name format.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RecordType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the resource, which indicates the type of value that Route 53 returns in response to DNS
     *         queries. You can specify values for <code>Type</code> in the following combinations:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>A</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>AAAA</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>SRV</code> </b>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CNAME</code> </b>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you want Cloud Map to create a Route 53 alias record when you register an instance, specify
     *         <code>A</code> or <code>AAAA</code> for <code>Type</code>.
     *         </p>
     *         <p>
     *         You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when
     *         you register an instance. For more information, see <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     *         </p>
     *         <p>
     *         The following values are supported:
     *         </p>
     *         <dl>
     *         <dt>A</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
     *         </p>
     *         </dd>
     *         <dt>AAAA</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the IP address of the resource in IPv6 format, such as
     *         2001:0db8:85a3:0000:0000:abcd:0001:2345.
     *         </p>
     *         </dd>
     *         <dt>CNAME</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specify the domain name that you want to route traffic to when you register an instance. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
     *         >Attributes</a> in the topic <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
     *         <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code> error.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>SRV</dt>
     *         <dd>
     *         <p>
     *         Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses
     *         the following values:
     *         </p>
     *         <p>
     *         <code>priority weight port service-hostname</code>
     *         </p>
     *         <p>
     *         Note the following about the values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be
     *         changed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>port</code> comes from the value that you specify for the
     *         <code>AWS_INSTANCE_PORT</code> attribute when you submit a <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
     *         request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>service-hostname</code> is a concatenation of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The value that you specify for <code>InstanceId</code> when you register an instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name of the service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name of the namespace.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
     *         <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
     *         <code>service-hostname</code> is the following:
     *         </p>
     *         <p>
     *         <code>test.backend.example.com</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify settings for an <code>SRV</code> record, note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
     *         <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
     *         <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
     *         <code>SRV</code> record. You can ignore these records.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a
     *         href=
     *         "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
     *         >Name</a> element in the documentation about <code>CreateService</code> for information about how to
     *         specify the correct name format.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     * @see RecordType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The amount of time, in seconds, that you want DNS resolvers to cache the settings for this record.
     * </p>
     * <note>
     * <p>
     * Alias records don't include a TTL because Route 53 uses the TTL for the Amazon Web Services resource that an
     * alias record routes traffic to. If you include the <code>AWS_ALIAS_DNS_NAME</code> attribute when you submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request,
     * the <code>TTL</code> value is ignored. Always specify a TTL for the service; you can use a service to register
     * instances that create either alias or non-alias records.
     * </p>
     * </note>
     * 
     * @return The amount of time, in seconds, that you want DNS resolvers to cache the settings for this record.</p>
     *         <note>
     *         <p>
     *         Alias records don't include a TTL because Route 53 uses the TTL for the Amazon Web Services resource that
     *         an alias record routes traffic to. If you include the <code>AWS_ALIAS_DNS_NAME</code> attribute when you
     *         submit a <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
     *         request, the <code>TTL</code> value is ignored. Always specify a TTL for the service; you can use a
     *         service to register instances that create either alias or non-alias records.
     *         </p>
     */
    public final Long ttl() {
        return ttl;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ttl());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DnsRecord)) {
            return false;
        }
        DnsRecord other = (DnsRecord) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(ttl(), other.ttl());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DnsRecord").add("Type", typeAsString()).add("TTL", ttl()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "TTL":
            return Optional.ofNullable(clazz.cast(ttl()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DnsRecord, T> g) {
        return obj -> g.apply((DnsRecord) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DnsRecord> {
        /**
         * <p>
         * The type of the resource, which indicates the type of value that Route 53 returns in response to DNS queries.
         * You can specify values for <code>Type</code> in the following combinations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>A</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>AAAA</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>SRV</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CNAME</code> </b>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you want Cloud Map to create a Route 53 alias record when you register an instance, specify <code>A</code>
         * or <code>AAAA</code> for <code>Type</code>.
         * </p>
         * <p>
         * You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when you
         * register an instance. For more information, see <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
         * </p>
         * <p>
         * The following values are supported:
         * </p>
         * <dl>
         * <dt>A</dt>
         * <dd>
         * <p>
         * Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
         * </p>
         * </dd>
         * <dt>AAAA</dt>
         * <dd>
         * <p>
         * Route 53 returns the IP address of the resource in IPv6 format, such as
         * 2001:0db8:85a3:0000:0000:abcd:0001:2345.
         * </p>
         * </dd>
         * <dt>CNAME</dt>
         * <dd>
         * <p>
         * Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You specify the domain name that you want to route traffic to when you register an instance. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
         * >Attributes</a> in the topic <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
         * <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code> error.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>SRV</dt>
         * <dd>
         * <p>
         * Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses the
         * following values:
         * </p>
         * <p>
         * <code>priority weight port service-hostname</code>
         * </p>
         * <p>
         * Note the following about the values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be
         * changed.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>port</code> comes from the value that you specify for the <code>AWS_INSTANCE_PORT</code>
         * attribute when you submit a <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>service-hostname</code> is a concatenation of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The value that you specify for <code>InstanceId</code> when you register an instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name of the service.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name of the namespace.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
         * <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
         * <code>service-hostname</code> is the following:
         * </p>
         * <p>
         * <code>test.backend.example.com</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify settings for an <code>SRV</code> record, note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
         * <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
         * <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
         * <code>SRV</code> record. You can ignore these records.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a href=
         * "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
         * >Name</a> element in the documentation about <code>CreateService</code> for information about how to specify
         * the correct name format.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param type
         *        The type of the resource, which indicates the type of value that Route 53 returns in response to DNS
         *        queries. You can specify values for <code>Type</code> in the following combinations:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>A</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>AAAA</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>SRV</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CNAME</code> </b>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you want Cloud Map to create a Route 53 alias record when you register an instance, specify
         *        <code>A</code> or <code>AAAA</code> for <code>Type</code>.
         *        </p>
         *        <p>
         *        You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records,
         *        when you register an instance. For more information, see <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html"
         *        >RegisterInstance</a>.
         *        </p>
         *        <p>
         *        The following values are supported:
         *        </p>
         *        <dl>
         *        <dt>A</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
         *        </p>
         *        </dd>
         *        <dt>AAAA</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the IP address of the resource in IPv6 format, such as
         *        2001:0db8:85a3:0000:0000:abcd:0001:2345.
         *        </p>
         *        </dd>
         *        <dt>CNAME</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You specify the domain name that you want to route traffic to when you register an instance. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
         *        >Attributes</a> in the topic <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html"
         *        >RegisterInstance</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
         *        <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code>
         *        error.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>SRV</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record
         *        uses the following values:
         *        </p>
         *        <p>
         *        <code>priority weight port service-hostname</code>
         *        </p>
         *        <p>
         *        Note the following about the values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't
         *        be changed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>port</code> comes from the value that you specify for the
         *        <code>AWS_INSTANCE_PORT</code> attribute when you submit a <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         *        request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>service-hostname</code> is a concatenation of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The value that you specify for <code>InstanceId</code> when you register an instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name of the service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name of the namespace.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
         *        <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
         *        <code>service-hostname</code> is the following:
         *        </p>
         *        <p>
         *        <code>test.backend.example.com</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify settings for an <code>SRV</code> record, note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in
         *        the <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
         *        <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
         *        <code>SRV</code> record. You can ignore these records.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a
         *        href=
         *        "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
         *        >Name</a> element in the documentation about <code>CreateService</code> for information about how to
         *        specify the correct name format.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @see RecordType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the resource, which indicates the type of value that Route 53 returns in response to DNS queries.
         * You can specify values for <code>Type</code> in the following combinations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>A</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>AAAA</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>SRV</code> </b>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CNAME</code> </b>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you want Cloud Map to create a Route 53 alias record when you register an instance, specify <code>A</code>
         * or <code>AAAA</code> for <code>Type</code>.
         * </p>
         * <p>
         * You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records, when you
         * register an instance. For more information, see <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
         * </p>
         * <p>
         * The following values are supported:
         * </p>
         * <dl>
         * <dt>A</dt>
         * <dd>
         * <p>
         * Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
         * </p>
         * </dd>
         * <dt>AAAA</dt>
         * <dd>
         * <p>
         * Route 53 returns the IP address of the resource in IPv6 format, such as
         * 2001:0db8:85a3:0000:0000:abcd:0001:2345.
         * </p>
         * </dd>
         * <dt>CNAME</dt>
         * <dd>
         * <p>
         * Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You specify the domain name that you want to route traffic to when you register an instance. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
         * >Attributes</a> in the topic <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
         * <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code> error.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>SRV</dt>
         * <dd>
         * <p>
         * Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record uses the
         * following values:
         * </p>
         * <p>
         * <code>priority weight port service-hostname</code>
         * </p>
         * <p>
         * Note the following about the values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't be
         * changed.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>port</code> comes from the value that you specify for the <code>AWS_INSTANCE_PORT</code>
         * attribute when you submit a <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>service-hostname</code> is a concatenation of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The value that you specify for <code>InstanceId</code> when you register an instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name of the service.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name of the namespace.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
         * <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
         * <code>service-hostname</code> is the following:
         * </p>
         * <p>
         * <code>test.backend.example.com</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify settings for an <code>SRV</code> record, note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in the
         * <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
         * <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
         * <code>SRV</code> record. You can ignore these records.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a href=
         * "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
         * >Name</a> element in the documentation about <code>CreateService</code> for information about how to specify
         * the correct name format.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param type
         *        The type of the resource, which indicates the type of value that Route 53 returns in response to DNS
         *        queries. You can specify values for <code>Type</code> in the following combinations:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>A</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>AAAA</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>A</code> </b> and <b> <code>AAAA</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>SRV</code> </b>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CNAME</code> </b>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you want Cloud Map to create a Route 53 alias record when you register an instance, specify
         *        <code>A</code> or <code>AAAA</code> for <code>Type</code>.
         *        </p>
         *        <p>
         *        You specify other settings, such as the IP address for <code>A</code> and <code>AAAA</code> records,
         *        when you register an instance. For more information, see <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html"
         *        >RegisterInstance</a>.
         *        </p>
         *        <p>
         *        The following values are supported:
         *        </p>
         *        <dl>
         *        <dt>A</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the IP address of the resource in IPv4 format, such as 192.0.2.44.
         *        </p>
         *        </dd>
         *        <dt>AAAA</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the IP address of the resource in IPv6 format, such as
         *        2001:0db8:85a3:0000:0000:abcd:0001:2345.
         *        </p>
         *        </dd>
         *        <dt>CNAME</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the domain name of the resource, such as www.example.com. Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You specify the domain name that you want to route traffic to when you register an instance. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html#cloudmap-RegisterInstance-request-Attributes"
         *        >Attributes</a> in the topic <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html"
         *        >RegisterInstance</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You must specify <code>WEIGHTED</code> for the value of <code>RoutingPolicy</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can't specify both <code>CNAME</code> for <code>Type</code> and settings for
         *        <code>HealthCheckConfig</code>. If you do, the request will fail with an <code>InvalidInput</code>
         *        error.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>SRV</dt>
         *        <dd>
         *        <p>
         *        Route 53 returns the value for an <code>SRV</code> record. The value for an <code>SRV</code> record
         *        uses the following values:
         *        </p>
         *        <p>
         *        <code>priority weight port service-hostname</code>
         *        </p>
         *        <p>
         *        Note the following about the values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The values of <code>priority</code> and <code>weight</code> are both set to <code>1</code> and can't
         *        be changed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>port</code> comes from the value that you specify for the
         *        <code>AWS_INSTANCE_PORT</code> attribute when you submit a <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         *        request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>service-hostname</code> is a concatenation of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The value that you specify for <code>InstanceId</code> when you register an instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name of the service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name of the namespace.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, if the value of <code>InstanceId</code> is <code>test</code>, the name of the service is
         *        <code>backend</code>, and the name of the namespace is <code>example.com</code>, the value of
         *        <code>service-hostname</code> is the following:
         *        </p>
         *        <p>
         *        <code>test.backend.example.com</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify settings for an <code>SRV</code> record, note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify values for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both in
         *        the <code>RegisterInstance</code> request, Cloud Map automatically creates <code>A</code> and/or
         *        <code>AAAA</code> records that have the same name as the value of <code>service-hostname</code> in the
         *        <code>SRV</code> record. You can ignore these records.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you're using a system that requires a specific <code>SRV</code> format, such as HAProxy, see the <a
         *        href=
         *        "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html#cloudmap-CreateService-request-Name"
         *        >Name</a> element in the documentation about <code>CreateService</code> for information about how to
         *        specify the correct name format.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @see RecordType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordType
         */
        Builder type(RecordType type);

        /**
         * <p>
         * The amount of time, in seconds, that you want DNS resolvers to cache the settings for this record.
         * </p>
         * <note>
         * <p>
         * Alias records don't include a TTL because Route 53 uses the TTL for the Amazon Web Services resource that an
         * alias record routes traffic to. If you include the <code>AWS_ALIAS_DNS_NAME</code> attribute when you submit
         * a <a href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         * request, the <code>TTL</code> value is ignored. Always specify a TTL for the service; you can use a service
         * to register instances that create either alias or non-alias records.
         * </p>
         * </note>
         * 
         * @param ttl
         *        The amount of time, in seconds, that you want DNS resolvers to cache the settings for this record.</p>
         *        <note>
         *        <p>
         *        Alias records don't include a TTL because Route 53 uses the TTL for the Amazon Web Services resource
         *        that an alias record routes traffic to. If you include the <code>AWS_ALIAS_DNS_NAME</code> attribute
         *        when you submit a <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a>
         *        request, the <code>TTL</code> value is ignored. Always specify a TTL for the service; you can use a
         *        service to register instances that create either alias or non-alias records.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ttl(Long ttl);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private Long ttl;

        private BuilderImpl() {
        }

        private BuilderImpl(DnsRecord model) {
            type(model.type);
            ttl(model.ttl);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(RecordType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Long getTtl() {
            return ttl;
        }

        public final void setTtl(Long ttl) {
            this.ttl = ttl;
        }

        @Override
        public final Builder ttl(Long ttl) {
            this.ttl = ttl;
            return this;
        }

        @Override
        public DnsRecord build() {
            return new DnsRecord(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
