/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspaces.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a WorkSpace image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkspaceImage implements SdkPojo, Serializable, ToCopyableBuilder<WorkspaceImage.Builder, WorkspaceImage> {
    private static final SdkField<String> IMAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageId").getter(getter(WorkspaceImage::imageId)).setter(setter(Builder::imageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(WorkspaceImage::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(WorkspaceImage::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<OperatingSystem> OPERATING_SYSTEM_FIELD = SdkField
            .<OperatingSystem> builder(MarshallingType.SDK_POJO).memberName("OperatingSystem")
            .getter(getter(WorkspaceImage::operatingSystem)).setter(setter(Builder::operatingSystem))
            .constructor(OperatingSystem::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(WorkspaceImage::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> REQUIRED_TENANCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequiredTenancy").getter(getter(WorkspaceImage::requiredTenancyAsString))
            .setter(setter(Builder::requiredTenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequiredTenancy").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(WorkspaceImage::errorCode)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(WorkspaceImage::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<Instant> CREATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Created").getter(getter(WorkspaceImage::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Created").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerAccountId").getter(getter(WorkspaceImage::ownerAccountId)).setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerAccountId").build()).build();

    private static final SdkField<UpdateResult> UPDATES_FIELD = SdkField.<UpdateResult> builder(MarshallingType.SDK_POJO)
            .memberName("Updates").getter(getter(WorkspaceImage::updates)).setter(setter(Builder::updates))
            .constructor(UpdateResult::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Updates").build()).build();

    private static final SdkField<List<ErrorDetails>> ERROR_DETAILS_FIELD = SdkField
            .<List<ErrorDetails>> builder(MarshallingType.LIST)
            .memberName("ErrorDetails")
            .getter(getter(WorkspaceImage::errorDetails))
            .setter(setter(Builder::errorDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorDetails").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ErrorDetails> builder(MarshallingType.SDK_POJO)
                                            .constructor(ErrorDetails::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_ID_FIELD, NAME_FIELD,
            DESCRIPTION_FIELD, OPERATING_SYSTEM_FIELD, STATE_FIELD, REQUIRED_TENANCY_FIELD, ERROR_CODE_FIELD,
            ERROR_MESSAGE_FIELD, CREATED_FIELD, OWNER_ACCOUNT_ID_FIELD, UPDATES_FIELD, ERROR_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String imageId;

    private final String name;

    private final String description;

    private final OperatingSystem operatingSystem;

    private final String state;

    private final String requiredTenancy;

    private final String errorCode;

    private final String errorMessage;

    private final Instant created;

    private final String ownerAccountId;

    private final UpdateResult updates;

    private final List<ErrorDetails> errorDetails;

    private WorkspaceImage(BuilderImpl builder) {
        this.imageId = builder.imageId;
        this.name = builder.name;
        this.description = builder.description;
        this.operatingSystem = builder.operatingSystem;
        this.state = builder.state;
        this.requiredTenancy = builder.requiredTenancy;
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
        this.created = builder.created;
        this.ownerAccountId = builder.ownerAccountId;
        this.updates = builder.updates;
        this.errorDetails = builder.errorDetails;
    }

    /**
     * <p>
     * The identifier of the image.
     * </p>
     * 
     * @return The identifier of the image.
     */
    public final String imageId() {
        return imageId;
    }

    /**
     * <p>
     * The name of the image.
     * </p>
     * 
     * @return The name of the image.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the image.
     * </p>
     * 
     * @return The description of the image.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The operating system that the image is running.
     * </p>
     * 
     * @return The operating system that the image is running.
     */
    public final OperatingSystem operatingSystem() {
        return operatingSystem;
    }

    /**
     * <p>
     * The status of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceImageState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of the image.
     * @see WorkspaceImageState
     */
    public final WorkspaceImageState state() {
        return WorkspaceImageState.fromValue(state);
    }

    /**
     * <p>
     * The status of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceImageState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of the image.
     * @see WorkspaceImageState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is enabled, this
     * value is set to <code>DEDICATED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own Windows
     * Desktop Images</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requiredTenancy}
     * will return {@link WorkspaceImageRequiredTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #requiredTenancyAsString}.
     * </p>
     * 
     * @return Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is
     *         enabled, this value is set to <code>DEDICATED</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
     *         Windows Desktop Images</a>.
     * @see WorkspaceImageRequiredTenancy
     */
    public final WorkspaceImageRequiredTenancy requiredTenancy() {
        return WorkspaceImageRequiredTenancy.fromValue(requiredTenancy);
    }

    /**
     * <p>
     * Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is enabled, this
     * value is set to <code>DEDICATED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own Windows
     * Desktop Images</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requiredTenancy}
     * will return {@link WorkspaceImageRequiredTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #requiredTenancyAsString}.
     * </p>
     * 
     * @return Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is
     *         enabled, this value is set to <code>DEDICATED</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
     *         Windows Desktop Images</a>.
     * @see WorkspaceImageRequiredTenancy
     */
    public final String requiredTenancyAsString() {
        return requiredTenancy;
    }

    /**
     * <p>
     * The error code that is returned for the image.
     * </p>
     * 
     * @return The error code that is returned for the image.
     */
    public final String errorCode() {
        return errorCode;
    }

    /**
     * <p>
     * The text of the error message that is returned for the image.
     * </p>
     * 
     * @return The text of the error message that is returned for the image.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * The date when the image was created. If the image has been shared, the Amazon Web Services account that the image
     * has been shared with sees the original creation date of the image.
     * </p>
     * 
     * @return The date when the image was created. If the image has been shared, the Amazon Web Services account that
     *         the image has been shared with sees the original creation date of the image.
     */
    public final Instant created() {
        return created;
    }

    /**
     * <p>
     * The identifier of the Amazon Web Services account that owns the image.
     * </p>
     * 
     * @return The identifier of the Amazon Web Services account that owns the image.
     */
    public final String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * <p>
     * The updates (if any) that are available for the specified image.
     * </p>
     * 
     * @return The updates (if any) that are available for the specified image.
     */
    public final UpdateResult updates() {
        return updates;
    }

    /**
     * For responses, this returns true if the service returned a value for the ErrorDetails property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasErrorDetails() {
        return errorDetails != null && !(errorDetails instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional details of the error returned for the image, including the possible causes of the errors and
     * troubleshooting information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasErrorDetails} method.
     * </p>
     * 
     * @return Additional details of the error returned for the image, including the possible causes of the errors and
     *         troubleshooting information.
     */
    public final List<ErrorDetails> errorDetails() {
        return errorDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystem());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requiredTenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(errorCode());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(updates());
        hashCode = 31 * hashCode + Objects.hashCode(hasErrorDetails() ? errorDetails() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkspaceImage)) {
            return false;
        }
        WorkspaceImage other = (WorkspaceImage) obj;
        return Objects.equals(imageId(), other.imageId()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description())
                && Objects.equals(operatingSystem(), other.operatingSystem())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(requiredTenancyAsString(), other.requiredTenancyAsString())
                && Objects.equals(errorCode(), other.errorCode()) && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(created(), other.created()) && Objects.equals(ownerAccountId(), other.ownerAccountId())
                && Objects.equals(updates(), other.updates()) && hasErrorDetails() == other.hasErrorDetails()
                && Objects.equals(errorDetails(), other.errorDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkspaceImage").add("ImageId", imageId()).add("Name", name()).add("Description", description())
                .add("OperatingSystem", operatingSystem()).add("State", stateAsString())
                .add("RequiredTenancy", requiredTenancyAsString()).add("ErrorCode", errorCode())
                .add("ErrorMessage", errorMessage()).add("Created", created()).add("OwnerAccountId", ownerAccountId())
                .add("Updates", updates()).add("ErrorDetails", hasErrorDetails() ? errorDetails() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageId":
            return Optional.ofNullable(clazz.cast(imageId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystem()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "RequiredTenancy":
            return Optional.ofNullable(clazz.cast(requiredTenancyAsString()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCode()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "Created":
            return Optional.ofNullable(clazz.cast(created()));
        case "OwnerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "Updates":
            return Optional.ofNullable(clazz.cast(updates()));
        case "ErrorDetails":
            return Optional.ofNullable(clazz.cast(errorDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkspaceImage, T> g) {
        return obj -> g.apply((WorkspaceImage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkspaceImage> {
        /**
         * <p>
         * The identifier of the image.
         * </p>
         * 
         * @param imageId
         *        The identifier of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageId(String imageId);

        /**
         * <p>
         * The name of the image.
         * </p>
         * 
         * @param name
         *        The name of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the image.
         * </p>
         * 
         * @param description
         *        The description of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The operating system that the image is running.
         * </p>
         * 
         * @param operatingSystem
         *        The operating system that the image is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * The operating system that the image is running.
         * </p>
         * This is a convenience method that creates an instance of the {@link OperatingSystem.Builder} avoiding the
         * need to create one manually via {@link OperatingSystem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OperatingSystem.Builder#build()} is called immediately and its
         * result is passed to {@link #operatingSystem(OperatingSystem)}.
         * 
         * @param operatingSystem
         *        a consumer that will call methods on {@link OperatingSystem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operatingSystem(OperatingSystem)
         */
        default Builder operatingSystem(Consumer<OperatingSystem.Builder> operatingSystem) {
            return operatingSystem(OperatingSystem.builder().applyMutation(operatingSystem).build());
        }

        /**
         * <p>
         * The status of the image.
         * </p>
         * 
         * @param state
         *        The status of the image.
         * @see WorkspaceImageState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceImageState
         */
        Builder state(String state);

        /**
         * <p>
         * The status of the image.
         * </p>
         * 
         * @param state
         *        The status of the image.
         * @see WorkspaceImageState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceImageState
         */
        Builder state(WorkspaceImageState state);

        /**
         * <p>
         * Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is enabled,
         * this value is set to <code>DEDICATED</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
         * Windows Desktop Images</a>.
         * </p>
         * 
         * @param requiredTenancy
         *        Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is
         *        enabled, this value is set to <code>DEDICATED</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your
         *        Own Windows Desktop Images</a>.
         * @see WorkspaceImageRequiredTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceImageRequiredTenancy
         */
        Builder requiredTenancy(String requiredTenancy);

        /**
         * <p>
         * Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is enabled,
         * this value is set to <code>DEDICATED</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
         * Windows Desktop Images</a>.
         * </p>
         * 
         * @param requiredTenancy
         *        Specifies whether the image is running on dedicated hardware. When Bring Your Own License (BYOL) is
         *        enabled, this value is set to <code>DEDICATED</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your
         *        Own Windows Desktop Images</a>.
         * @see WorkspaceImageRequiredTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceImageRequiredTenancy
         */
        Builder requiredTenancy(WorkspaceImageRequiredTenancy requiredTenancy);

        /**
         * <p>
         * The error code that is returned for the image.
         * </p>
         * 
         * @param errorCode
         *        The error code that is returned for the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * The text of the error message that is returned for the image.
         * </p>
         * 
         * @param errorMessage
         *        The text of the error message that is returned for the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * The date when the image was created. If the image has been shared, the Amazon Web Services account that the
         * image has been shared with sees the original creation date of the image.
         * </p>
         * 
         * @param created
         *        The date when the image was created. If the image has been shared, the Amazon Web Services account
         *        that the image has been shared with sees the original creation date of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(Instant created);

        /**
         * <p>
         * The identifier of the Amazon Web Services account that owns the image.
         * </p>
         * 
         * @param ownerAccountId
         *        The identifier of the Amazon Web Services account that owns the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * The updates (if any) that are available for the specified image.
         * </p>
         * 
         * @param updates
         *        The updates (if any) that are available for the specified image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updates(UpdateResult updates);

        /**
         * <p>
         * The updates (if any) that are available for the specified image.
         * </p>
         * This is a convenience method that creates an instance of the {@link UpdateResult.Builder} avoiding the need
         * to create one manually via {@link UpdateResult#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UpdateResult.Builder#build()} is called immediately and its
         * result is passed to {@link #updates(UpdateResult)}.
         * 
         * @param updates
         *        a consumer that will call methods on {@link UpdateResult.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #updates(UpdateResult)
         */
        default Builder updates(Consumer<UpdateResult.Builder> updates) {
            return updates(UpdateResult.builder().applyMutation(updates).build());
        }

        /**
         * <p>
         * Additional details of the error returned for the image, including the possible causes of the errors and
         * troubleshooting information.
         * </p>
         * 
         * @param errorDetails
         *        Additional details of the error returned for the image, including the possible causes of the errors
         *        and troubleshooting information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorDetails(Collection<ErrorDetails> errorDetails);

        /**
         * <p>
         * Additional details of the error returned for the image, including the possible causes of the errors and
         * troubleshooting information.
         * </p>
         * 
         * @param errorDetails
         *        Additional details of the error returned for the image, including the possible causes of the errors
         *        and troubleshooting information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorDetails(ErrorDetails... errorDetails);

        /**
         * <p>
         * Additional details of the error returned for the image, including the possible causes of the errors and
         * troubleshooting information.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.workspaces.model.ErrorDetails.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.workspaces.model.ErrorDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.workspaces.model.ErrorDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #errorDetails(List<ErrorDetails>)}.
         * 
         * @param errorDetails
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.workspaces.model.ErrorDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errorDetails(java.util.Collection<ErrorDetails>)
         */
        Builder errorDetails(Consumer<ErrorDetails.Builder>... errorDetails);
    }

    static final class BuilderImpl implements Builder {
        private String imageId;

        private String name;

        private String description;

        private OperatingSystem operatingSystem;

        private String state;

        private String requiredTenancy;

        private String errorCode;

        private String errorMessage;

        private Instant created;

        private String ownerAccountId;

        private UpdateResult updates;

        private List<ErrorDetails> errorDetails = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(WorkspaceImage model) {
            imageId(model.imageId);
            name(model.name);
            description(model.description);
            operatingSystem(model.operatingSystem);
            state(model.state);
            requiredTenancy(model.requiredTenancy);
            errorCode(model.errorCode);
            errorMessage(model.errorMessage);
            created(model.created);
            ownerAccountId(model.ownerAccountId);
            updates(model.updates);
            errorDetails(model.errorDetails);
        }

        public final String getImageId() {
            return imageId;
        }

        public final void setImageId(String imageId) {
            this.imageId = imageId;
        }

        @Override
        public final Builder imageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final OperatingSystem.Builder getOperatingSystem() {
            return operatingSystem != null ? operatingSystem.toBuilder() : null;
        }

        public final void setOperatingSystem(OperatingSystem.BuilderImpl operatingSystem) {
            this.operatingSystem = operatingSystem != null ? operatingSystem.build() : null;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(WorkspaceImageState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getRequiredTenancy() {
            return requiredTenancy;
        }

        public final void setRequiredTenancy(String requiredTenancy) {
            this.requiredTenancy = requiredTenancy;
        }

        @Override
        public final Builder requiredTenancy(String requiredTenancy) {
            this.requiredTenancy = requiredTenancy;
            return this;
        }

        @Override
        public final Builder requiredTenancy(WorkspaceImageRequiredTenancy requiredTenancy) {
            this.requiredTenancy(requiredTenancy == null ? null : requiredTenancy.toString());
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final Instant getCreated() {
            return created;
        }

        public final void setCreated(Instant created) {
            this.created = created;
        }

        @Override
        public final Builder created(Instant created) {
            this.created = created;
            return this;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final UpdateResult.Builder getUpdates() {
            return updates != null ? updates.toBuilder() : null;
        }

        public final void setUpdates(UpdateResult.BuilderImpl updates) {
            this.updates = updates != null ? updates.build() : null;
        }

        @Override
        public final Builder updates(UpdateResult updates) {
            this.updates = updates;
            return this;
        }

        public final List<ErrorDetails.Builder> getErrorDetails() {
            List<ErrorDetails.Builder> result = ErrorDetailsListCopier.copyToBuilder(this.errorDetails);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setErrorDetails(Collection<ErrorDetails.BuilderImpl> errorDetails) {
            this.errorDetails = ErrorDetailsListCopier.copyFromBuilder(errorDetails);
        }

        @Override
        public final Builder errorDetails(Collection<ErrorDetails> errorDetails) {
            this.errorDetails = ErrorDetailsListCopier.copy(errorDetails);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errorDetails(ErrorDetails... errorDetails) {
            errorDetails(Arrays.asList(errorDetails));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errorDetails(Consumer<ErrorDetails.Builder>... errorDetails) {
            errorDetails(Stream.of(errorDetails).map(c -> ErrorDetails.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public WorkspaceImage build() {
            return new WorkspaceImage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
