/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.xray.model.BatchGetTracesRequest;
import software.amazon.awssdk.services.xray.model.BatchGetTracesResponse;
import software.amazon.awssdk.services.xray.model.CreateGroupRequest;
import software.amazon.awssdk.services.xray.model.CreateGroupResponse;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.DeleteGroupRequest;
import software.amazon.awssdk.services.xray.model.DeleteGroupResponse;
import software.amazon.awssdk.services.xray.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.xray.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.GetGroupRequest;
import software.amazon.awssdk.services.xray.model.GetGroupResponse;
import software.amazon.awssdk.services.xray.model.GetGroupsRequest;
import software.amazon.awssdk.services.xray.model.GetGroupsResponse;
import software.amazon.awssdk.services.xray.model.GetInsightEventsRequest;
import software.amazon.awssdk.services.xray.model.GetInsightEventsResponse;
import software.amazon.awssdk.services.xray.model.GetInsightImpactGraphRequest;
import software.amazon.awssdk.services.xray.model.GetInsightImpactGraphResponse;
import software.amazon.awssdk.services.xray.model.GetInsightRequest;
import software.amazon.awssdk.services.xray.model.GetInsightResponse;
import software.amazon.awssdk.services.xray.model.GetInsightSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetInsightSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsResponse;
import software.amazon.awssdk.services.xray.model.GetServiceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetServiceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse;
import software.amazon.awssdk.services.xray.model.GetTraceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetTraceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse;
import software.amazon.awssdk.services.xray.model.InvalidPolicyRevisionIdException;
import software.amazon.awssdk.services.xray.model.InvalidRequestException;
import software.amazon.awssdk.services.xray.model.ListResourcePoliciesRequest;
import software.amazon.awssdk.services.xray.model.ListResourcePoliciesResponse;
import software.amazon.awssdk.services.xray.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.xray.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.xray.model.LockoutPreventionException;
import software.amazon.awssdk.services.xray.model.MalformedPolicyDocumentException;
import software.amazon.awssdk.services.xray.model.PolicyCountLimitExceededException;
import software.amazon.awssdk.services.xray.model.PolicySizeLimitExceededException;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.xray.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsRequest;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsResponse;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsRequest;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsResponse;
import software.amazon.awssdk.services.xray.model.ResourceNotFoundException;
import software.amazon.awssdk.services.xray.model.RuleLimitExceededException;
import software.amazon.awssdk.services.xray.model.TagResourceRequest;
import software.amazon.awssdk.services.xray.model.TagResourceResponse;
import software.amazon.awssdk.services.xray.model.ThrottledException;
import software.amazon.awssdk.services.xray.model.TooManyTagsException;
import software.amazon.awssdk.services.xray.model.UntagResourceRequest;
import software.amazon.awssdk.services.xray.model.UntagResourceResponse;
import software.amazon.awssdk.services.xray.model.UpdateGroupRequest;
import software.amazon.awssdk.services.xray.model.UpdateGroupResponse;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.XRayException;
import software.amazon.awssdk.services.xray.transform.BatchGetTracesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightEventsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightImpactGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingRulesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingStatisticSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingTargetsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetServiceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTimeSeriesServiceStatisticsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.ListResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTelemetryRecordsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTraceSegmentsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link XRayClient}.
 *
 * @see XRayClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultXRayClient implements XRayClient {
    private static final Logger log = Logger.loggerFor(DefaultXRayClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final XRayServiceClientConfiguration serviceClientConfiguration;

    protected DefaultXRayClient(XRayServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     *
     * @param batchGetTracesRequest
     * @return Result of the BatchGetTraces operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.BatchGetTraces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetTracesResponse batchGetTraces(BatchGetTracesRequest batchGetTracesRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetTracesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetTracesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetTracesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetTraces");

            return clientHandler.execute(new ClientExecutionParams<BatchGetTracesRequest, BatchGetTracesResponse>()
                    .withOperationName("BatchGetTraces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchGetTracesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetTracesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a group resource with a name and a filter expression.
     * </p>
     *
     * @param createGroupRequest
     * @return Result of the CreateGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                    .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a rule to control sampling behavior for instrumented applications. Services retrieve rules with <a
     * href="https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html">GetSamplingRules</a>, and evaluate
     * each rule in ascending order of <i>priority</i> for each request. If a rule matches, the service records a trace,
     * borrowing it from the reservoir size. After 10 seconds, the service reports back to X-Ray with <a
     * href="https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html">GetSamplingTargets</a> to get
     * updated versions of each in-use rule. The updated rule contains a trace quota that the service can use instead of
     * borrowing from the reservoir.
     * </p>
     *
     * @param createSamplingRuleRequest
     * @return Result of the CreateSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws RuleLimitExceededException
     *         You have reached the maximum number of sampling rules.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.CreateSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSamplingRuleResponse createSamplingRule(CreateSamplingRuleRequest createSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, RuleLimitExceededException, AwsServiceException,
            SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSamplingRule");

            return clientHandler.execute(new ClientExecutionParams<CreateSamplingRuleRequest, CreateSamplingRuleResponse>()
                    .withOperationName("CreateSamplingRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSamplingRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSamplingRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a group resource.
     * </p>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                    .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a resource policy from the target Amazon Web Services account.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws InvalidPolicyRevisionIdException
     *         A policy revision id was provided which does not match the latest policy revision. This exception is also
     *         if a policy revision id of 0 is provided via <code>PutResourcePolicy</code> and a policy with the same
     *         name already exists.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws InvalidRequestException, InvalidPolicyRevisionIdException, ThrottledException, AwsServiceException,
            SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a sampling rule.
     * </p>
     *
     * @param deleteSamplingRuleRequest
     * @return Result of the DeleteSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.DeleteSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSamplingRuleResponse deleteSamplingRule(DeleteSamplingRuleRequest deleteSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSamplingRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteSamplingRuleRequest, DeleteSamplingRuleResponse>()
                    .withOperationName("DeleteSamplingRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSamplingRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSamplingRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the current encryption configuration for X-Ray data.
     * </p>
     *
     * @param getEncryptionConfigRequest
     * @return Result of the GetEncryptionConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetEncryptionConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEncryptionConfigResponse getEncryptionConfig(GetEncryptionConfigRequest getEncryptionConfigRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEncryptionConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEncryptionConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEncryptionConfig");

            return clientHandler.execute(new ClientExecutionParams<GetEncryptionConfigRequest, GetEncryptionConfigResponse>()
                    .withOperationName("GetEncryptionConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEncryptionConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEncryptionConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves group resource details.
     * </p>
     *
     * @param getGroupRequest
     * @return Result of the GetGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupResponse getGroup(GetGroupRequest getGroupRequest) throws InvalidRequestException, ThrottledException,
            AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroup");

            return clientHandler.execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>()
                    .withOperationName("GetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     *
     * @param getGroupsRequest
     * @return Result of the GetGroups operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupsResponse getGroups(GetGroupsRequest getGroupsRequest) throws InvalidRequestException, ThrottledException,
            AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroups");

            return clientHandler.execute(new ClientExecutionParams<GetGroupsRequest, GetGroupsResponse>()
                    .withOperationName("GetGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the summary information of an insight. This includes impact to clients and root cause services, the top
     * anomalous services, the category, the state of the insight, and the start and end time of the insight.
     * </p>
     *
     * @param getInsightRequest
     * @return Result of the GetInsight operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInsightResponse getInsight(GetInsightRequest getInsightRequest) throws InvalidRequestException, ThrottledException,
            AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsight");

            return clientHandler
                    .execute(new ClientExecutionParams<GetInsightRequest, GetInsightResponse>().withOperationName("GetInsight")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getInsightRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * X-Ray reevaluates insights periodically until they're resolved, and records each intermediate state as an event.
     * You can review an insight's events in the Impact Timeline on the Inspect page in the X-Ray console.
     * </p>
     *
     * @param getInsightEventsRequest
     * @return Result of the GetInsightEvents operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetInsightEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInsightEventsResponse getInsightEvents(GetInsightEventsRequest getInsightEventsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInsightEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightEvents");

            return clientHandler.execute(new ClientExecutionParams<GetInsightEventsRequest, GetInsightEventsResponse>()
                    .withOperationName("GetInsightEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInsightEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a service graph structure filtered by the specified insight. The service graph is limited to only
     * structural information. For a complete service graph, use this API with the GetServiceGraph API.
     * </p>
     *
     * @param getInsightImpactGraphRequest
     * @return Result of the GetInsightImpactGraph operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetInsightImpactGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightImpactGraph" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetInsightImpactGraphResponse getInsightImpactGraph(GetInsightImpactGraphRequest getInsightImpactGraphRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightImpactGraphResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInsightImpactGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightImpactGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightImpactGraph");

            return clientHandler.execute(new ClientExecutionParams<GetInsightImpactGraphRequest, GetInsightImpactGraphResponse>()
                    .withOperationName("GetInsightImpactGraph").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInsightImpactGraphRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightImpactGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the summaries of all insights in the specified group matching the provided filter values.
     * </p>
     *
     * @param getInsightSummariesRequest
     * @return Result of the GetInsightSummaries operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetInsightSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInsightSummariesResponse getInsightSummaries(GetInsightSummariesRequest getInsightSummariesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInsightSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightSummaries");

            return clientHandler.execute(new ClientExecutionParams<GetInsightSummariesRequest, GetInsightSummariesResponse>()
                    .withOperationName("GetInsightSummaries").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInsightSummariesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return Result of the GetSamplingRules operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSamplingRulesResponse getSamplingRules(GetSamplingRulesRequest getSamplingRulesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSamplingRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSamplingRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingRules");

            return clientHandler.execute(new ClientExecutionParams<GetSamplingRulesRequest, GetSamplingRulesResponse>()
                    .withOperationName("GetSamplingRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSamplingRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return Result of the GetSamplingStatisticSummaries operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingStatisticSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSamplingStatisticSummariesResponse getSamplingStatisticSummaries(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingStatisticSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSamplingStatisticSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSamplingStatisticSummariesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingStatisticSummaries");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSamplingStatisticSummariesRequest, GetSamplingStatisticSummariesResponse>()
                            .withOperationName("GetSamplingStatisticSummaries").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingStatisticSummariesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSamplingStatisticSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Requests a sampling quota for rules that the service is using to sample requests.
     * </p>
     *
     * @param getSamplingTargetsRequest
     * @return Result of the GetSamplingTargets operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetSamplingTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSamplingTargetsResponse getSamplingTargets(GetSamplingTargetsRequest getSamplingTargetsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSamplingTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSamplingTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSamplingTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingTargets");

            return clientHandler.execute(new ClientExecutionParams<GetSamplingTargetsRequest, GetSamplingTargetsResponse>()
                    .withOperationName("GetSamplingTargets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSamplingTargetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSamplingTargetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">Amazon Web Services X-Ray
     * SDK</a>. Downstream services can be other applications, Amazon Web Services resources, HTTP web APIs, or SQL
     * databases.
     * </p>
     *
     * @param getServiceGraphRequest
     * @return Result of the GetServiceGraph operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetServiceGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetServiceGraphResponse getServiceGraph(GetServiceGraphRequest getServiceGraphRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceGraph");

            return clientHandler.execute(new ClientExecutionParams<GetServiceGraphRequest, GetServiceGraphResponse>()
                    .withOperationName("GetServiceGraph").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getServiceGraphRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServiceGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return Result of the GetTimeSeriesServiceStatistics operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTimeSeriesServiceStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTimeSeriesServiceStatisticsResponse getTimeSeriesServiceStatistics(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTimeSeriesServiceStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTimeSeriesServiceStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTimeSeriesServiceStatisticsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTimeSeriesServiceStatistics");

            return clientHandler
                    .execute(new ClientExecutionParams<GetTimeSeriesServiceStatisticsRequest, GetTimeSeriesServiceStatisticsResponse>()
                            .withOperationName("GetTimeSeriesServiceStatistics").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getTimeSeriesServiceStatisticsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetTimeSeriesServiceStatisticsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     *
     * @param getTraceGraphRequest
     * @return Result of the GetTraceGraph operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceGraphResponse getTraceGraph(GetTraceGraphRequest getTraceGraphRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTraceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTraceGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTraceGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTraceGraph");

            return clientHandler.execute(new ClientExecutionParams<GetTraceGraphRequest, GetTraceGraphResponse>()
                    .withOperationName("GetTraceGraph").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTraceGraphRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTraceGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>Amazon Web Services X-Ray Developer Guide</i>.
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return Result of the GetTraceSummaries operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.GetTraceSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTraceSummariesResponse getTraceSummaries(GetTraceSummariesRequest getTraceSummariesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTraceSummariesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTraceSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTraceSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTraceSummaries");

            return clientHandler.execute(new ClientExecutionParams<GetTraceSummariesRequest, GetTraceSummariesResponse>()
                    .withOperationName("GetTraceSummaries").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTraceSummariesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTraceSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of resource policies in the target Amazon Web Services account.
     * </p>
     *
     * @param listResourcePoliciesRequest
     * @return Result of the ListResourcePolicies operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.ListResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/ListResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcePoliciesResponse listResourcePolicies(ListResourcePoliciesRequest listResourcePoliciesRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourcePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcePolicies");

            return clientHandler.execute(new ClientExecutionParams<ListResourcePoliciesRequest, ListResourcePoliciesResponse>()
                    .withOperationName("ListResourcePolicies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourcePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags that are applied to the specified Amazon Web Services X-Ray group or sampling rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws ResourceNotFoundException
     *         The resource was not found. Verify that the name or Amazon Resource Name (ARN) of the resource is
     *         correct.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidRequestException, ThrottledException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the encryption configuration for X-Ray data.
     * </p>
     *
     * @param putEncryptionConfigRequest
     * @return Result of the PutEncryptionConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutEncryptionConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutEncryptionConfigResponse putEncryptionConfig(PutEncryptionConfigRequest putEncryptionConfigRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutEncryptionConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEncryptionConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEncryptionConfig");

            return clientHandler.execute(new ClientExecutionParams<PutEncryptionConfigRequest, PutEncryptionConfigResponse>()
                    .withOperationName("PutEncryptionConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putEncryptionConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutEncryptionConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the resource policy to grant one or more Amazon Web Services services and accounts permissions to access
     * X-Ray. Each resource policy will be associated with a specific Amazon Web Services account. Each Amazon Web
     * Services account can have a maximum of 5 resource policies, and each policy name must be unique within that
     * account. The maximum size of each resource policy is 5KB.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws MalformedPolicyDocumentException
     *         Invalid policy document provided in request.
     * @throws LockoutPreventionException
     *         The provided resource policy would prevent the caller of this request from calling PutResourcePolicy in
     *         the future.
     * @throws InvalidPolicyRevisionIdException
     *         A policy revision id was provided which does not match the latest policy revision. This exception is also
     *         if a policy revision id of 0 is provided via <code>PutResourcePolicy</code> and a policy with the same
     *         name already exists.
     * @throws PolicySizeLimitExceededException
     *         Exceeded the maximum size for a resource policy.
     * @throws PolicyCountLimitExceededException
     *         Exceeded the maximum number of resource policies for a target Amazon Web Services account.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws MalformedPolicyDocumentException, LockoutPreventionException, InvalidPolicyRevisionIdException,
            PolicySizeLimitExceededException, PolicyCountLimitExceededException, ThrottledException, AwsServiceException,
            SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used by the Amazon Web Services X-Ray daemon to upload telemetry.
     * </p>
     *
     * @param putTelemetryRecordsRequest
     * @return Result of the PutTelemetryRecords operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutTelemetryRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTelemetryRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutTelemetryRecordsResponse putTelemetryRecords(PutTelemetryRecordsRequest putTelemetryRecordsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutTelemetryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutTelemetryRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTelemetryRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTelemetryRecords");

            return clientHandler.execute(new ClientExecutionParams<PutTelemetryRecordsRequest, PutTelemetryRecordsResponse>()
                    .withOperationName("PutTelemetryRecords").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putTelemetryRecordsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutTelemetryRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads segment documents to Amazon Web Services X-Ray. The <a
     * href="https://docs.aws.amazon.com/xray/index.html">X-Ray SDK</a> generates segment documents and sends them to
     * the X-Ray daemon, which uploads them in batches. A segment document can be a completed segment, an in-progress
     * segment, or an array of subsegments.
     * </p>
     * <p>
     * Segments must include the following fields. For the full segment document schema, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-api-segmentdocuments.html">Amazon Web Services X-Ray
     * Segment Documents</a> in the <i>Amazon Web Services X-Ray Developer Guide</i>.
     * </p>
     * <p class="title">
     * <b>Required segment document fields</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>name</code> - The name of the service that handled the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>id</code> - A 64-bit identifier for the segment, unique among segments in the same trace, in 16 hexadecimal
     * digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>trace_id</code> - A unique identifier that connects all segments and subsegments originating from a single
     * client request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>start_time</code> - Time the segment or subsegment was created, in floating point seconds in epoch time,
     * accurate to milliseconds. For example, <code>1480615200.010</code> or <code>1.480615200010E9</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>end_time</code> - Time the segment or subsegment was closed. For example, <code>1480615200.090</code> or
     * <code>1.480615200090E9</code>. Specify either an <code>end_time</code> or <code>in_progress</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>in_progress</code> - Set to <code>true</code> instead of specifying an <code>end_time</code> to record that
     * a segment has been started, but is not complete. Send an in-progress segment when your application receives a
     * request that will take a long time to serve, to trace that the request was received. When the response is sent,
     * send the complete segment to overwrite the in-progress segment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A <code>trace_id</code> consists of three numbers separated by hyphens. For example,
     * 1-58406520-a006649127e371903a2de979. This includes:
     * </p>
     * <p class="title">
     * <b>Trace ID Format</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The version number, for instance, <code>1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The time of the original request, in Unix epoch time, in 8 hexadecimal digits. For example, 10:00AM December 2nd,
     * 2016 PST in epoch time is <code>1480615200</code> seconds, or <code>58406520</code> in hexadecimal.
     * </p>
     * </li>
     * <li>
     * <p>
     * A 96-bit identifier for the trace, globally unique, in 24 hexadecimal digits.
     * </p>
     * </li>
     * </ul>
     *
     * @param putTraceSegmentsRequest
     * @return Result of the PutTraceSegments operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.PutTraceSegments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTraceSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutTraceSegmentsResponse putTraceSegments(PutTraceSegmentsRequest putTraceSegmentsRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutTraceSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutTraceSegmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTraceSegmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTraceSegments");

            return clientHandler.execute(new ClientExecutionParams<PutTraceSegmentsRequest, PutTraceSegmentsResponse>()
                    .withOperationName("PutTraceSegments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putTraceSegmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutTraceSegmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies tags to an existing Amazon Web Services X-Ray group or sampling rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws ResourceNotFoundException
     *         The resource was not found. Verify that the name or Amazon Resource Name (ARN) of the resource is
     *         correct.
     * @throws TooManyTagsException
     *         You have exceeded the maximum number of tags you can apply to this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidRequestException,
            ThrottledException, ResourceNotFoundException, TooManyTagsException, AwsServiceException, SdkClientException,
            XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Web Services X-Ray group or sampling rule. You cannot edit or delete system tags
     * (those with an <code>aws:</code> prefix).
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws ResourceNotFoundException
     *         The resource was not found. Verify that the name or Amazon Resource Name (ARN) of the resource is
     *         correct.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidRequestException,
            ThrottledException, ResourceNotFoundException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a group resource.
     * </p>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws InvalidRequestException,
            ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                    .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies a sampling rule's configuration.
     * </p>
     *
     * @param updateSamplingRuleRequest
     * @return Result of the UpdateSamplingRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is missing required parameters or has invalid parameters.
     * @throws ThrottledException
     *         The request exceeds the maximum number of requests per second.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws XRayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample XRayClient.UpdateSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSamplingRuleResponse updateSamplingRule(UpdateSamplingRuleRequest updateSamplingRuleRequest)
            throws InvalidRequestException, ThrottledException, AwsServiceException, SdkClientException, XRayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSamplingRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSamplingRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateSamplingRuleRequest, UpdateSamplingRuleResponse>()
                    .withOperationName("UpdateSamplingRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSamplingRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSamplingRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(XRayException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException")
                                .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException")
                                .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RuleLimitExceededException")
                                .exceptionBuilderSupplier(RuleLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyCountLimitExceededException")
                                .exceptionBuilderSupplier(PolicyCountLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockoutPreventionException")
                                .exceptionBuilderSupplier(LockoutPreventionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException")
                                .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final XRayServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
